# coding: utf-8
from __future__ import absolute_import, division, print_function, unicode_literals

from collections import namedtuple
from logging import getLogger

import six
import sqlalchemy
from sqlalchemy.dialects.postgresql import JSONB

from travel.rasp.bus.db import session_scope
from travel.rasp.bus.db.models.order import Order
from travel.rasp.bus.registry.mail_sender import mail_sender
from travel.rasp.bus.scripts.order_report.utils import dump_csv, get_human_month

logger = getLogger("order_report")


class ReportBillingRow(
    namedtuple(
        "ReportBillingRow",
        "trust_refund_id timestamp service_product_id service_order_id developer_payload amount currency",
    )
):
    def __new__(cls, **kwargs):
        return super(ReportBillingRow, cls).__new__(cls, *(kwargs.get(field, "") for field in cls._fields))


def _get_basket_item(purchase_data, service_order_id):
    return next(item_data for item_data in purchase_data["orders"] if item_data["service_order_id"] == service_order_id)


def gen_billing_rows(billing_log_entry, purchase_data):
    entry_type = billing_log_entry["type"]

    if entry_type == "payment":
        for payment_order in billing_log_entry["orders"]:
            yield ReportBillingRow(
                timestamp=billing_log_entry["timestamp"],
                service_product_id=payment_order["service_product_id"],
                service_order_id=payment_order["service_order_id"],
                developer_payload=payment_order["developer_payload"],
                amount=payment_order["amount"],
                currency=payment_order["currency"],
            )
        return

    if entry_type == "refund":
        if billing_log_entry.get("orders_unknown"):
            yield ReportBillingRow(
                timestamp=billing_log_entry["timestamp"],
                trust_refund_id=billing_log_entry["trust_refund_id"],
                amount=billing_log_entry["amount"],
            )
            return

        for refund_order in billing_log_entry["orders"]:
            service_order_id = refund_order["service_order_id"]
            payment_order = _get_basket_item(purchase_data, service_order_id)
            yield ReportBillingRow(
                timestamp=billing_log_entry["timestamp"],
                trust_refund_id=billing_log_entry["trust_refund_id"],
                service_product_id=payment_order["service_product_id"],
                service_order_id=service_order_id,
                developer_payload=payment_order["developer_payload"],
                amount="-{}".format(refund_order["amount"]),
                currency=refund_order["currency"],
            )


def dump_detailed_report(year, month, limit=None):
    def gen_rows():
        yield [
            "order_id",
            "creation_ts",
            "booking_supplier_id",
            "trust_payment_id",
            "trust_refund_id",
            "timestamp",
            "service_product_id",
            "service_order_id",
            "developer_payload",
            "amount",
            "currency",
        ]

        with session_scope() as session:
            billing_log_q = sqlalchemy.func.jsonb_array_elements(Order.billing_log).alias("billing_log")
            billing_log_column = sqlalchemy.column("billing_log.value", is_literal=True, type_=JSONB)

            for (order_id, creation_ts, booking_supplier_id, purchase, billing_log_entry) in (
                session.query(
                    Order.id, Order.creation_ts, Order.booking["supplierId"], Order.purchase, billing_log_column
                )
                .select_from(billing_log_q)
                .filter(billing_log_column["timestamp"].astext.startswith("{}-{:02}-".format(year, month)))
                .order_by(billing_log_column["timestamp"])
                .limit(limit)
            ):
                order_columns = [
                    order_id,
                    "" if creation_ts is None else six.text_type(creation_ts.isoformat()),
                    booking_supplier_id,
                    purchase["trust_payment_id"],
                ]

                for billing_row in gen_billing_rows(billing_log_entry, purchase):
                    yield order_columns + list(billing_row)

    return dump_csv(gen_rows())


def detailed_report(year, month, emails):
    logger.info("Creating the detailed order report for %d year and month %d", year, month)

    csv_report = dump_detailed_report(year, month)
    human_month = get_human_month(month)
    mail_sender.send_email(
        to_emails=emails,
        subject="Yandex.Bus: Подробный месячный отчет за {} {} года".format(human_month, year),
        body="""
Добрый день!

<p>Во вложении к этому письму находится подробный месячный отчет об обороте сервиса Яндекс.Автобусы за {} {} года.</p>

С уважением, команда Яндекс.Автобусов.
""".format(
            human_month, year
        ),
        files=[("detailed-report-{}-{:02d}.csv".format(year, month), csv_report)],
    )
    logger.info("Finished creating the detailed order report for %d year and month %d", year, month)
