# coding: utf-8
from __future__ import absolute_import, division, print_function, unicode_literals

import logging

import msgpack

from travel.library.python.dicts.settlement_repository import SettlementRepository
from travel.library.python.dicts.station_repository import StationRepository
from travel.library.python.dicts.station_to_settlement_repository import StationToSettlementRepository
from travel.rasp.bus.db import session_scope
from travel.rasp.bus.db.models.matching import PointMatching, PointType, parse_point_key


log = logging.getLogger()


class PointKeyFilterGenerator(object):
    """
    FilterData generator for go_suggests
    BUSES-171
    """

    TRANSPORT_TYPES = {
        'train': 1,
        'plane': 2,
        'bus': 3,
        'river': 4,
        'sea': 5,
        'suburban': 6,
        'helicopter': 7,
        'urban': 8,
        'psevdo_gortrans': 9,
        'blablacar': 10,
        'water': 11
    }

    @staticmethod
    def _get_matching_point_keys(station_repo, station_to_settlement_repo):
        point_keys = set()
        with session_scope() as session:
            for point_key in session.query(PointMatching.point_key).distinct():
                point_key_type, point_key_id = parse_point_key(point_key.point_key)
                if point_key_type is not PointType.INVALID:
                    point_keys.add(point_key.point_key)
                if point_key_type is not PointType.STATION:
                    continue
                settlement = station_repo.get(point_key_id)
                if settlement is not None:
                    point_keys.add('{}{}'.format(PointType.CITY.point_key_prefix, settlement.SettlementId))
                station_to_settlement = station_to_settlement_repo.get(point_key_id)
                if station_to_settlement is not None:
                    point_keys.add('{}{}'.format(PointType.CITY.point_key_prefix, station_to_settlement.SettlementId))

        return point_keys

    @staticmethod
    def _get_ttype_point_keys(station_repo, settlement_repo, ttype):
        point_keys = set()
        if ttype not in PointKeyFilterGenerator.TRANSPORT_TYPES:
            log.error("PointKeyFilterGenerator.gen_ttype() invalid ttype value: {}".format(ttype))
            raise ValueError("invalid ttype value")

        ttype_id = PointKeyFilterGenerator.TRANSPORT_TYPES[ttype]
        for station in station_repo.itervalues():
            if station.TransportType == ttype_id:
                point_keys.add('{}{}'.format(PointType.STATION.point_key_prefix, station.Id))

        for settlement in settlement_repo.itervalues():
            if settlement.Majority < 5:  # city+
                point_keys.add('{}{}'.format(PointType.CITY.point_key_prefix, settlement.Id))

        return point_keys

    @staticmethod
    def gen(station_fn, pointkey_fn, station_to_settlement_fn=None, settlement_fn=None, ttype=None):
        """
        Method generates suggests filter of matched points.
        If ttype is not None - method generates suggests filter using specified transport type
        ----------
        station_fn : str
            Station dump filename
        station_to_settlement_fn : str
            Station to settlement dump filename
        settlement_fn : str
            Settlement dump filename
        pointkey_fn :
            Filter output filename
        ttype : str
            The code of the transport type. See PointKeyFilterGenerator.TRANSPORT_TYPES
        """
        station_repo = StationRepository()
        station_repo.load_from_file(station_fn)

        if ttype:
            settlement_repo = SettlementRepository()
            settlement_repo.load_from_file(settlement_fn)
            log.info('Repo loaded')
            point_keys = PointKeyFilterGenerator._get_ttype_point_keys(station_repo, settlement_repo, ttype)
        else:
            station_to_settlement_repo = StationToSettlementRepository(index_field='StationId')
            station_to_settlement_repo.load_from_file(station_to_settlement_fn)
            log.info('Repo loaded')
            point_keys = PointKeyFilterGenerator._get_matching_point_keys(station_repo, station_to_settlement_repo)

        with open(pointkey_fn, 'wb') as fd:
            msgpack.pack({'PointKeys': {pk: True for pk in point_keys}}, fd)
        log.info('Created FilterData. Dumped %s pointkeys', len(point_keys))
