import logging

import psycopg2
import psycopg2.extras
import requests

from travel.rasp.bus.scripts.refund_checker.src.bus_client import (
    get_order_by_id,
    get_refund_check_data_by_order_id,
)
from travel.rasp.bus.scripts.refund_checker.src.config import Config


def _validate(fiscal_urls, refund_ids, ticket_statuses):
    if not fiscal_urls:
        return "Can not find any fiscal url"
    if len(fiscal_urls) != 1:
        return "Fiscal url is not unique"
    if not refund_ids:
        return "Can not find any refund id"
    if not ticket_statuses:
        return "Order does not have any ticket"
    if not all(s == 'returned' for s in ticket_statuses):
        return "Some tickets was not refunded"
    return None


def _build_refund_check_url(fiscal_url, refund_id):
    url, query = fiscal_url.split('?')
    url_parts = url.split('/')
    url_parts[-1] = refund_id
    return '/'.join(url_parts) + '?' + query


def _download_reciept(refund_check_url):
    r = requests.get(refund_check_url)
    if r.status_code != 200:
        return False
    return True


def check(order_id):
    logging.info('Let\'s check that order with [%s] was returned\n', order_id)
    with psycopg2.connect(Config.CONNECTION_STRING) as db_connection:
        order = get_order_by_id(db_connection, order_id)
        if order is None:
            logging.error("Can't find order")
            return False
        ticket_statuses = [t['status']['name'] for t in order['booking']['tickets']]
        fiscal_urls, refund_ids = get_refund_check_data_by_order_id(db_connection, order_id)

    logging.info("Common information about this refund:")
    logging.info("\tOrder id : %s", order_id)
    logging.info("\tUser passport uid : %s", order['uid'])
    logging.info("\tOrder url : https://bus.yandex.ru/admin/orders/%s", order_id)
    logging.info("\tLogs url : https://bus.yandex.ru/admin/orders/%s/log", order_id)
    logging.info("\tTicket statuses:")
    for i, ticket in enumerate(order['booking']['tickets']):
        logging.info("\t\t%d) %s", i+1, ticket['status']['name'])
    logging.info("\tFiscal urls:")
    for i, url in enumerate(fiscal_urls):
        logging.info("\t\t%d) %s", i+1, url)
    logging.info("\tRefund ids:")
    for i, refund_id in enumerate(refund_ids):
        logging.info("\t\t%d) %s", i+1, refund_id)

    error_message = _validate(fiscal_urls, refund_ids, ticket_statuses)
    if error_message is not None:
        logging.error(error_message)
        return False

    fiscal_url = list(fiscal_urls)[0]
    refund_check_urls = [
        _build_refund_check_url(fiscal_url, refund_id)
        for refund_id in refund_ids
    ]
    logging.info("""\tRefund reciept urls:""")
    for i, url in enumerate(refund_check_urls):
        logging.info("\t\t%d) %s", i+1, url)
    logging.info("")

    logging.info("""Let's try to fetch the refund reciepts:""")
    ok_reciept_count = 0
    for url in refund_check_urls:
        is_ok = _download_reciept(url)
        if is_ok:
            logging.info('\tReceipt url [%s] is correct', url)
            ok_reciept_count += 1
        else:
            logging.warn('\tReceipt url [%s] is not correct', url)
    logging.info("")

    if len(ticket_statuses) != ok_reciept_count:
        logging.warn('Some tickets were not refund. Expected %d refunds. Actual %d refunds',
                     len(ticket_statuses), ok_reciept_count)
        return False
    logging.info('All is right!')
    return True
