# coding: utf-8
from __future__ import absolute_import, division, print_function, unicode_literals

import logging

import boto3
import requests
from requests.adapters import HTTPAdapter
from requests.packages.urllib3.util.retry import Retry

from travel.library.python.rasp_vault.api import get_secret


log = logging.getLogger(__name__)


def retry_session(retries=5, backoff_factor=1, status_forcelist=(500, 502, 504), session=None):
    session = session or requests.Session()
    retry = Retry(
        total=retries,
        read=retries,
        connect=retries,
        backoff_factor=backoff_factor,
        status_forcelist=status_forcelist,
    )
    adapter = HTTPAdapter(max_retries=retry)
    session.mount('https://', adapter)
    return session


class Sitemap:

    S3_ENDPOINT = 'https://s3.mds.yandex.net'
    S3_BUCKET = 'yandex-bus'
    S3_PREFIX = 'sitemap/'
    S3_ACCESS_KEY = 'sec-01d956rm0wknnrbhhjsyx0rh56.robot-sputnik-s3-mds-key'
    S3_ACCESS_SECRET_KEY = 'sec-01d956rm0wknnrbhhjsyx0rh56.robot-sputnik-s3-mds-secret'
    SITEMAP_FN = 'sitemap.txt'

    POPULAR_DIRECTIONS = 'https://backend.internal.bus.yandex.net/api/popular?limit={}'

    BASE_URL = 'https://travel.yandex.ru/buses/'
    URL_RIDE_TMPL = BASE_URL + '{}--{}/'
    URL_CITY_TMPL = BASE_URL + '{}/'

    dry = None
    threshold = None
    popular_limit = None

    def __init__(self, popular_limit, count_threshold, dry=False):
        self.dry = dry
        self.threshold = count_threshold
        self.popular_limit = popular_limit
        s3_key = get_secret(self.S3_ACCESS_KEY)
        s3_secret_key = get_secret(self.S3_ACCESS_SECRET_KEY)
        session = boto3.session.Session(
            aws_access_key_id=s3_key,
            aws_secret_access_key=s3_secret_key,
        )
        self.s3 = session.client(service_name='s3', endpoint_url=self.S3_ENDPOINT, verify=False)

    def run(self):
        if self.dry:
            log.info('Dry run!')
        log.info('run with popular_limit=%d, threshold=%d', self.popular_limit, self.threshold)
        log.info('getting popular directions')
        populars = self._get_popular_directions()
        populars_count = len(populars)
        log.info('got popular directions number: %d', populars_count)
        if populars_count < self.threshold:
            raise ValueError('upload restricted. popular count below threshold, need at least {}, got {}'.format(
                self.threshold, populars_count))
        log.info('generating sitemap')
        self._generate_sitemap(populars)
        if self.dry:
            log.info('Dry run! Sitemap upload is restricted')
        else:
            log.info('uploading sitemap file to S3')
            is_ok, response = self._upload_sitemap()
            if not is_ok:
                raise ValueError('error uploading file to S3, got result: %s', response)
            log.info('success upload sitemap file to S3')

    def _upload_sitemap(self):
        with open(self.SITEMAP_FN, 'rb') as fb:
            response = self.s3.put_object(
                ACL='public-read',
                Body=fb,
                Bucket=self.S3_BUCKET,
                Key=self.S3_PREFIX + self.SITEMAP_FN,
                ContentType='application/octet-stream'
            )
        if response['ResponseMetadata']['HTTPStatusCode'] == 200:
            return True, None
        else:
            return False, response

    def _get_popular_directions(self):
        session = retry_session()
        raw_populars = session.get(self.POPULAR_DIRECTIONS.format(self.popular_limit)).json()
        populars = raw_populars.get('Directions') or []
        return populars

    def _generate_sitemap(self, populars):
        sitemap = [self.BASE_URL + '\n']
        cities = set()
        for popular in populars:
            from_slug = popular['From'].get('Slug')
            to_slug = popular['To'].get('Slug')
            if not from_slug or not to_slug:
                continue
            sitemap.append(self.URL_RIDE_TMPL.format(from_slug, to_slug) + '\n')
            cities.add(self.URL_CITY_TMPL.format(from_slug) + '\n')

        sitemap.extend(cities)
        with open(self.SITEMAP_FN, 'w') as fout:
            fout.writelines(sitemap)
