import logging
import re
from contextlib import contextmanager
from functools import partial

import zeep
from lxml import etree
from zeep.cache import SqliteCache, InMemoryCache
from zeep.transports import Transport

from travel.rasp.bus.spark_api import models

log = logging.getLogger(__name__)


FIO_PATTERN = re.compile(r'(.+?)\s+(.+?)\.\s*(.+?)\.')

ENVIRONMENT_TO_HOST = {
    'production': 'webservicefarm.interfax.ru',
    'testing': 'sparkgatetest.interfax.ru',
}

IN_MEMORY_CACHE = object()


class SparkException(Exception):
    pass


class SparkAuthError(SparkException):
    pass


class SparkConfig:
    def __init__(self, host, login, password, cache_db=None):
        self.host = host
        self.login = login
        self.password = password
        self.cache_db = cache_db

    def __repr__(self):
        return '<SparkConfig host={} login={} password=******* cache_db={}>'.format(
            self.host,
            self.login,
            self.cache_db,
        )


class NoResult(Exception):
    pass


class SparkClient:
    def __init__(self, config, session):
        self._config = config
        cache = None
        if config.cache_db is IN_MEMORY_CACHE:
            cache = InMemoryCache(timeout=None)
        elif config.cache_db is not None:
            cache = SqliteCache(path=config.cache_db, timeout=None)

        transport = Transport(session=session, cache=cache)
        self._client = zeep.Client(
            'http://{}//IfaxWebService/?wsdl'.format(config.host),
            transport=transport,
        )
        self._service = self._client.service

    def _call(self, method, **params):
        result = self._service[method](**params)
        if result[f'{method}Result'] != 'True':
            raise NoResult

        root = etree.XML(result.xmlData.encode())
        log.debug(
            '%s xmlData:\n%s',
            method,
            etree.tostring(root, encoding='utf-8', pretty_print=True).decode()
        )

        return root

    def list_regions(self):
        return sorted(
            (
                models.Region(code=elem.findtext('Code'), name=elem.findtext('NameRus'))
                for elem in self._call('GetListRegion').iterfind('Data/Classifier/Elem')
            ),
            key=lambda r: r.code
        )

    def list_region_codes(self):
        return tuple(r.code for r in self.list_regions())

    def ifind_companies_by_name(self, company_name, region_codes=None, okopf_code=None):
        if region_codes is None:
            region_codes = self.list_region_codes()

        for region_code in region_codes:
            try:
                root = self._call(
                    'GetCompanyListByName',
                    companyName=company_name, regionCode=region_code, exactSearch='1', egrpoIncluded='2'
                )
            except NoResult:
                continue

            for xml_company in root.iterfind('Data/Company'):
                actual_okopf_code = xml_company.findtext('OKOPFCode')
                if okopf_code is not None and okopf_code != actual_okopf_code:
                    continue
                yield models.Company(
                    id=int(xml_company.findtext('SparkID') or '0'),
                    inn=xml_company.findtext('INN'),
                    ogrn=xml_company.findtext('OGRN'),
                    okpo=xml_company.findtext('OKPO'),
                    full_name=xml_company.findtext('FullName'),
                    address=xml_company.findtext('Address'),
                    industry=xml_company.findtext('Industry'),
                    okopf_name=xml_company.findtext('OKOPFName'),
                    okopf_code=xml_company.findtext('OKOPFCode'),
                    manager=xml_company.findtext('Manager'),
                )

    def find_companies_by_name(self, company_name, region_codes=None, okopf_code=None):
        return tuple(self.ifind_companies_by_name(company_name, region_codes, okopf_code))

    @staticmethod
    def _entrepreneur_full_name_predicate(full_name, first_name_letter, middle_name_letter):
        full_name_parts = full_name.strip().lower().split()

        if len(full_name_parts) == 3:
            _last_name, first_name, middle_name = full_name_parts
            return first_name[0] == first_name_letter and middle_name[0] == middle_name_letter

        if len(full_name_parts) == 2:
            _last_name, first_name = full_name_parts
            return first_name[0] == first_name_letter

        return True

    @classmethod
    def _make_entrepreneur_filter(cls, fio):
        fio = fio.strip().lower()
        fio_match = FIO_PATTERN.match(fio)
        if fio_match is None:
            return fio, lambda entrepreneur_fio: True

        last_name, first_name_letter, middle_name_letter = fio_match.groups()
        return last_name, partial(
            cls._entrepreneur_full_name_predicate,
            first_name_letter=first_name_letter,
            middle_name_letter=middle_name_letter
        )

    def ifind_entrepreneurs_by_name(self, fio, region_codes=None):
        if region_codes is None:
            region_codes = self.list_region_codes()

        query_fio, full_name_predicate = self._make_entrepreneur_filter(fio)
        for region_code in region_codes:
            try:
                root = self._call(
                    'GetEntrepreneurListByFIO',
                    fio=query_fio, isActing='1', regionCode=region_code
                )
            except NoResult:
                continue

            for xml_entrepreneur in root.iterfind('Data/Entrepreneur'):
                full_name = xml_entrepreneur.findtext('FullName')
                if full_name_predicate(full_name):
                    yield models.Entrepreneur(
                        id=int(xml_entrepreneur.findtext('SparkID') or '0'),
                        inn=xml_entrepreneur.findtext('INN'),
                        ogrnip=xml_entrepreneur.findtext('OGRNIP'),
                        okpo=xml_entrepreneur.findtext('OKPO'),
                        full_name=full_name,
                    )

    def find_entrepreneurs_by_name(self, fio, region_codes=None):
        return tuple(self.ifind_entrepreneurs_by_name(fio, region_codes))

    @staticmethod
    def _parse_okveds(xml_report):
        return tuple(
            models.OkvedItem(
                code=okved.get('Code', ''),
                is_main=okved.get('IsMain') == 'true',
                name=okved.get('Name', ''),
            )
            for okved in xml_report.iterfind('OKVED2List/OKVED')
        )

    def get_company_report(self, spark_id, inn, ogrn):
        try:
            root = self._call('GetCompanyExtendedReport', sparkId=spark_id, inn=inn, ogrn=ogrn)
        except NoResult:
            return None

        xml_report = root.find('Data/Report')
        if xml_report.find('Status').get('IsActing') != '1':
            return None

        xml_legal_address = xml_report.find('LegalAddresses/Address')
        xml_region = xml_report.find('OKATO')
        return models.CompanyReport(
            legal_address=xml_legal_address.get('Address', '') if xml_legal_address is not None else '',
            okveds=self._parse_okveds(xml_report),
            region=models.Region(
                code=xml_region.get('RegionCode'),
                name=xml_region.get('RegionName'),
            ),
            short_name=xml_report.findtext('ShortNameRus')
        )

    def get_entrepreneur_report(self, inn, ogrnip):
        try:
            root = self._call('GetEntrepreneurShortReport', inn=inn, ogrnip=ogrnip)
        except NoResult:
            return None

        xml_report = root.find('Data/Report')
        if xml_report.find('Status').get('IsActing') != '1':
            return None

        xml_region = xml_report.find('OKATO')
        return models.EntrepreneurReport(
            okveds=self._parse_okveds(xml_report),
            region=models.Region(
                code=xml_region.get('RegionCode'),
                name=xml_region.get('RegionName'),
            )
        )

    def find_entrepreneur_by_code(self, code):
        try:
            root = self._call('FindEntrepreneurByCode', entrepreneurCode=code)
        except NoResult:
            return None

        xml_entrepreneur = root.find('Data/Entrepreneur')
        return models.Entrepreneur(
            id=int(xml_entrepreneur.findtext('SparkID') or '0'),
            inn=xml_entrepreneur.findtext('INN'),
            ogrnip=xml_entrepreneur.findtext('OGRNIP'),
            okpo=xml_entrepreneur.findtext('OKPO'),
            full_name=xml_entrepreneur.findtext('FullName')
        )

    def find_company_by_code(self, code):
        try:
            root = self._call('FindCompanyByCode', companyCode=code, excludeBranch='1')
        except NoResult:
            return None

        xml_company = root.find('Data/Company')
        return models.Company(
            id=int(xml_company.findtext('SparkID') or '0'),
            inn=xml_company.findtext('INN'),
            ogrn=xml_company.findtext('OGRN'),
            okpo=xml_company.findtext('OKPO'),
            full_name=xml_company.findtext('FullName'),
            address=xml_company.findtext('Address'),
            industry=xml_company.findtext('Industry'),
            okopf_name=xml_company.findtext('OKOPFName'),
            okopf_code=xml_company.findtext('OKOPFCode'),
            manager=xml_company.findtext('Manager'),
        )

    def _start_session(self):
        result = self._service.Authmethod(Login=self._config.login, Password=self._config.password)
        if result != 'True':
            raise SparkAuthError('Password or login is not correct')

    def _end_session(self):
        self._service.End()

    @classmethod
    @contextmanager
    def create(cls, config, session=None):
        client = cls(config, session)
        try:
            client._start_session()
            yield client
        finally:
            client._end_session()
