import json
import logging
import logging.config

import gunicorn
import psycopg2
import psycopg2.extras
from flask import Flask, jsonify, request

from travel.rasp.bus_gdpr_api.src.bus_client import get_orders_by_uid
from travel.rasp.bus_gdpr_api.src.config import Config
from travel.rasp.bus_gdpr_api.src.tvm_client_store import tvm_client_store

flask_app = Flask('bus_gdpr_api')
flask_app.config.from_object('travel.rasp.bus_gdpr_api.src.config.Config')


def has_access():
    if Config.DISABLE_TVM:
        return True

    try:
        st = tvm_client_store.get().check_service_ticket(request.headers.get('X-Ya-Service-Ticket', ""))
        allow = st.src in Config.ALLOWED_CONSUMERS
        if not allow:
            logging.warn("The consumer with id [%s] is not allowed", st.src)

        return allow
    except Exception:
        logging.exception('TvmException')
        return False


@flask_app.route('/api/gdpr', methods=['POST', 'GET'])
def gpdr():
    if not has_access():
        return jsonify({
            "status": "error",
            "message": "access forbidden",
        })

    uid = None
    unixtime = None
    if request.method == 'POST':
        uid = request.form.get('uid')
        unixtime = request.form.get('unixtime')
    else:
        uid = request.args.get('uid')
        unixtime = request.args.get('unixtime')

    logging.info('Fetch data for %s in %s', uid, unixtime)

    try:
        with psycopg2.connect(Config.CONNECTION_STRING) as conn:
            orders = get_orders_by_uid(conn, uid)
            if not orders:
                logging.info('Can not find anything for %s in %s', uid, unixtime)
                return jsonify({
                    "status": "no_data",
                })

            logging.info('Can find something for %s in %s', uid, unixtime)
            return jsonify({
                "status": "ok",
                "data": {
                    'orders.json': json.dumps(orders),
                }
            })
    except Exception:
        logging.exception('ERROR /api/gdpr %s %s', uid, unixtime)
        return jsonify({
            "status": "error",
            "message": "internal error",
        })


@flask_app.route('/ping')
def ping():
    return 'ok'


class GunicornApp(gunicorn.app.base.Application):
    def __init__(self, flask_app):
        self.flask_app = flask_app
        super(GunicornApp, self).__init__()

    def init(self, parser, opts, args):
        pass

    def load(self):
        return self.flask_app


if __name__ == '__main__':
    logging.config.dictConfig(Config.LOGGING)
    logging.info("Start gunicorn")
    try:
        app = GunicornApp(
            flask_app,
        )
        app.run()
    except Exception as e:
        logging.exception(e)
        raise
