#!/usr/bin/env python

import getpass
import os
import subprocess
import tempfile

import click

_template = '''---
- hosts: {host}
  sudo: true
  gather_facts: yes
  remote_user: {remote_user}

  roles:
    - projects/{role}
'''


@click.group()
def roles():
    pass


@roles.command('apply_role', context_settings=dict(
    ignore_unknown_options=True,
))
@click.argument('role')
@click.argument('host')
@click.option('--remote_user')
@click.option('--use_root')
@click.argument('extra', nargs=-1, type=click.UNPROCESSED)
def apply_role(role, host, remote_user=None, use_root=False, extra=None):
    user = getpass.getuser()
    if not remote_user:
        remote_user = user
    if use_root:
        remote_user = 'root'
        work_dir_path = '/root/work'
    else:
        work_dir_path = '/home/{}/work'.format(remote_user)
    fd = tempfile.NamedTemporaryFile(dir='.')
    click.echo('creating temporary playbook at {}'.format(fd.name))
    try:
        fd.write(_template.format(host=host, role=role, remote_user=remote_user))
        fd.flush()
        command_args = [
            'ansible-playbook', '-i', '{},'.format(host), fd.name,
            '--extra-vars="@cloud/mysql_vars.yml"',
            '-e', 'remote_user={}'.format(remote_user),
            '-e', 'git_user={}'.format(user),
            '-e', 'work_dir_path={}'.format(work_dir_path)
        ] + (list(extra) if extra else [])
        print('running command:')
        print(command_args)
        subprocess.check_call(command_args)
    except Exception:
        import traceback
        traceback.print_exc()
        raise
    finally:
        os.close(fd)


@roles.command('list_roles')
def list_roles():
    stdout = subprocess.check_output(['ls', 'roles/projects'])
    click.echo(stdout.strip())


if __name__ == '__main__':
    roles()
