package main

import (
	"context"
	stdLog "log"
	"os"
	"sync"

	"a.yandex-team.ru/library/go/core/log"
	"a.yandex-team.ru/library/go/maxprocs"
	"a.yandex-team.ru/travel/library/go/configuration"
	"a.yandex-team.ru/travel/library/go/metrics"
	metricserver "a.yandex-team.ru/travel/library/go/metrics/server"
	"a.yandex-team.ru/travel/library/go/tracing"
	"a.yandex-team.ru/travel/library/go/vault"
	"a.yandex-team.ru/travel/rasp/crosslink/internal"
	"a.yandex-team.ru/travel/rasp/crosslink/internal/server"
	"a.yandex-team.ru/travel/rasp/crosslink/internal/storage"
)

const (
	RaspCrosslinkTesting    = "sec-01cjvsaf2hrdkfdr97c30dv3t4"
	RaspCrosslinkProduction = "sec-01cjvsaf2hrdkfdr97c30dv3t4"
)

func getSecUID() string {
	envType := os.Getenv("YENV_TYPE")
	secUID := ""
	switch envType {
	case "production":
		secUID = RaspCrosslinkProduction
	case "development":
		fallthrough
	case "testing":
		secUID = RaspCrosslinkTesting
	}
	return secUID
}

type Config struct {
	Addr         string `config:"http-addr"`
	GraphVersion int64  `config:"CROSSLINK_GRAPH_VERSION"`
	Metrics      metricserver.MetricsConfig
	Logging      internal.LogConfig
}

var DefaultConfig = Config{
	Addr:         "[::]:9000",
	GraphVersion: 0,
	Metrics:      metricserver.DefaultMetricsConfig,
	Logging:      internal.DefaultLoggingConfig,
}

func main() {
	maxprocs.AdjustAuto()
	configLoader := configuration.NewDefaultConfitaLoader()
	config := DefaultConfig
	_ = configLoader.Load(context.Background(), &config)

	storageConfig := storage.Parameters{
		PreparedSessionsCount: 3,
	}
	_ = configLoader.Load(context.Background(), &storageConfig)

	logger, err := internal.New(config.Logging)
	if err != nil {
		stdLog.Fatalf("can't set up logger: %v", err)
	}

	tracerCloser := tracing.InitializeDefaultTracer("rasp.crosslink")
	defer func() {
		err := tracerCloser.Close()
		if err != nil {
			logger.Error("tracer close error:", log.Error(err))
		}
	}()

	rootRegistry := metrics.NewRegistryWithDeployTags()

	if storageConfig.Token == "" {
		sr := vault.NewYavSecretsResolverFromEnv("RASP_VAULT_OAUTH_TOKEN")
		secUID := getSecUID()
		secret, err := sr.GetSecretValue(secUID, "yql-token")
		if err != nil {
			logger.Error("can't get token from vault: %v", log.Error(err))
		}
		storageConfig.Token = secret
	}

	wg := sync.WaitGroup{}
	wg.Add(2)

	go func() {
		defer wg.Done()

		err = server.Run(&storageConfig, logger, config.GraphVersion, config.Addr, rootRegistry.WithPrefix("http"))
		if err != nil {
			logger.Fatal("Error while starting application server", log.Error(err))
		}
	}()

	go func() {
		defer wg.Done()
		err = metricserver.RunMetricsHTTPServer(context.Background(), config.Metrics, logger, rootRegistry)
		if err != nil {
			logger.Fatal("Error while starting metrics server", log.Error(err))
		}
	}()

	logger.Info("Started")
	wg.Wait()

}
