package server

import (
	"context"
	"encoding/json"
	"fmt"
	"net/http"
	"strconv"

	"a.yandex-team.ru/library/go/core/log"
	"a.yandex-team.ru/library/go/core/metrics"
	"a.yandex-team.ru/library/go/httputil/middleware/httpmetrics"
	"a.yandex-team.ru/travel/library/go/tracing"
	"github.com/go-chi/chi/v5"
	"github.com/go-chi/chi/v5/middleware"

	"a.yandex-team.ru/travel/rasp/crosslink/internal/storage"
)

type Application struct {
	crosslinkRepository storage.CrosslinksRepository
	logger              log.Logger
	graphVersion        int64
}

func (app *Application) Ping(w http.ResponseWriter, r *http.Request) {
	app.logger.Info("request URL: ", log.Any("url", r.URL))
	w.WriteHeader(200)
	_, _ = w.Write([]byte("pong"))
}

func (app *Application) GetCrosslinks(w http.ResponseWriter, r *http.Request) {
	q := r.URL.Query()
	app.logger.Info("request URL: ", log.Any("url", r.URL))
	from := q.Get("from_key")
	if from == "" {
		w.WriteHeader(400)
		app.logger.Error("no from key")
		return
	}
	to := q.Get("to_key")
	if to == "" {
		w.WriteHeader(400)
		app.logger.Error("no to key")
		return
	}

	version := app.graphVersion
	var err error
	if q.Get("graph_version") != "" {
		version, err = strconv.ParseInt(q.Get("graph_version"), 10, 64)
		if err != nil {
			w.WriteHeader(400)
			app.logger.Errorf("incorrect graph version format: %+v", err)
			return
		}
	}

	transportType := q.Get("transport_type")
	if transportType == "" {
		transportType = "train"
	}

	crosslinks, err := app.crosslinkRepository.GetCrosslinks(r.Context(), from, to, version, transportType)
	if err != nil {
		w.WriteHeader(400)
		app.logger.Errorf("can't select crosslinks: %+v", err)
		return
	}
	bytes, err := json.Marshal(crosslinks)
	if err != nil {
		w.WriteHeader(400)
		app.logger.Errorf("can't marshal: %+v", err)
		return
	}
	app.logger.Info("response: ", log.ByteString("response", bytes))
	w.WriteHeader(200)
	_, _ = w.Write(bytes)
}

func BuildApplication(ctx context.Context, params *storage.Parameters,
	logger log.Logger, graphVersion int64) (*Application, error) {
	repo := storage.CrosslinksRepository{}
	ydbConfig := storage.NewYdbConfig(params)
	err := repo.Connect(ctx, params.Endpoint, ydbConfig, params.PreparedSessionsCount)
	if err != nil {
		return nil, fmt.Errorf("connecting YDB repository with params %+v and YdbConfig %+v: %w", params, ydbConfig, err)
	}
	application := &Application{
		crosslinkRepository: repo,
		logger:              logger,
		graphVersion:        graphVersion,
	}
	return application, nil
}

func Run(params *storage.Parameters, logger log.Logger, graphVersion int64, addr string, httpMetricsRegistry metrics.Registry) error {
	r := chi.NewRouter()
	r.Use(middleware.RequestID)
	r.Use(middleware.Logger)
	r.Use(httpmetrics.New(httpMetricsRegistry, httpmetrics.WithPathEndpoint(), httpmetrics.WithSolomonRated()))

	ctx, cancel := context.WithCancel(context.Background())
	defer cancel()

	app, err := BuildApplication(ctx, params, logger, graphVersion)
	if err != nil {
		return fmt.Errorf("error while initializing application: %w", err)

	}
	defer func() {
		err = app.crosslinkRepository.Close(ctx)
		if err != nil {
			logger.Error("Error while deinitializing repository", log.Error(err))
		}
	}()

	tm := tracing.NewTracingMiddlewareBuilder().
		WithFilter(tracing.NewPathFilter("/ping")).
		WithExtractor(tracing.NewHeaderTagExtractor("X-Request-Id", "request-id")).
		Build()
	r.Use(tm.Handle)

	r.Get("/crosslinks", app.GetCrosslinks)
	r.Get("/ping", app.Ping)

	return http.ListenAndServe(addr, r)
}
