package storage

import (
	"context"
	"fmt"
	"strings"
	"time"

	"a.yandex-team.ru/kikimr/public/sdk/go/ydb"
	"a.yandex-team.ru/kikimr/public/sdk/go/ydb/table"
	"github.com/opentracing/opentracing-go"
)

type Parameters struct {
	Endpoint              string `config:"CROSSLINK_YDB_ENDPOINT,required"`
	Database              string `config:"CROSSLINK_YDB_DATABASE,required"`
	Token                 string `config:"YDB_TOKEN"`
	PreparedSessionsCount int    `config:"YDB_PREPARED_SESSIONS_COUNT"`
}

type Crosslink struct {
	ToKey                 string `json:"toKey"`
	ToSlug                string `json:"toSlug"`
	ToTitleRuGenitive     string `json:"toTitleRuGenitive"`
	ToTitleRuAccusative   string `json:"toTitleRuAccusative"`
	ToTitleRuNominative   string `json:"toTitleRuNominative"`
	FromKey               string `json:"fromKey"`
	FromSlug              string `json:"fromSlug"`
	FromTitleRuGenitive   string `json:"fromTitleRuGenitive"`
	FromTitleRuAccusative string `json:"fromTitleRuAccusative"`
	FromTitleRuNominative string `json:"fromTitleRuNominative"`
}

type CrosslinksRepository struct {
	sessionPool           *table.SessionPool
	client                *table.Client
	preparedSessionsCount int
	readTx                *table.TransactionControl
	retryer               *table.Retryer
}

var Query = `
		DECLARE $transport_type AS Utf8;
		DECLARE $source_to_key AS Utf8;
		DECLARE $source_from_key AS Utf8;
		DECLARE $graph_version AS Int64;

		SELECT
		p1.title as from,
		p2.title as to,
		a.from_key as fromKey,
		a.to_key as toKey,
		p1.slug as fromSlug,
		p1.title_ru_genitive as fromTitleRuGenitive,
		p1.title_ru_accusative as fromTitleRuAccusative,
		p2.slug as toSlug,
		p2.title_ru_genitive as toTitleRuGenitive,
		p2.title_ru_accusative as toTitleRuAccusative
	FROM associates as a
	JOIN points as p1 ON p1.key = a.from_key
	JOIN points as p2 ON p2.key = a.to_key
	WHERE a.graph_version = $graph_version AND
		a.source_from_key = $source_from_key AND
		a.source_to_key = $source_to_key AND
		a.transport_type = $transport_type;`

func NewYdbConfig(params *Parameters) *ydb.DriverConfig {
	ydbConfig := new(ydb.DriverConfig)

	ydbConfig.Database = params.Database
	ydbConfig.Credentials = ydb.AuthTokenCredentials{
		AuthToken: params.Token,
	}
	return ydbConfig
}

func (repo *CrosslinksRepository) Connect(
	ctx context.Context,
	endpoint string,
	ydbConfig *ydb.DriverConfig,
	preparedSessionsCount int,
) error {
	driver, err := (&ydb.Dialer{
		DriverConfig: ydbConfig,
	}).Dial(ctx, endpoint)

	if err != nil {
		return err
	}
	repo.readTx = table.TxControl(
		table.BeginTx(
			table.WithOnlineReadOnly(),
		),
		table.CommitTx(),
	)
	repo.preparedSessionsCount = preparedSessionsCount
	repo.client = &table.Client{Driver: driver}
	repo.sessionPool = &table.SessionPool{
		IdleThreshold:      time.Second,
		Builder:            repo.client,
		SizeLimit:          3,
		KeepAliveBatchSize: -1,
	}
	repo.retryer = &table.Retryer{
		MaxRetries:      1,
		Backoff:         ydb.BackoffFunc(func(n int) <-chan time.Time { return time.After(0) }),
		SessionProvider: repo.sessionPool,
		RetryChecker:    ydb.RetryChecker{RetryNotFound: false},
	}
	return repo.fillSessionPool()
}

func (repo *CrosslinksRepository) Close(ctx context.Context) error {
	if err := repo.sessionPool.Close(ctx); err != nil {
		return err
	}
	return repo.client.Driver.Close()
}

func (repo *CrosslinksRepository) GetCrosslinks(ctx context.Context,
	sourceFromKey, sourceToKey string, graphVersion int64, transportType string) ([]Crosslink, error) {
	crosslinkGetterSpan, _ := opentracing.StartSpanFromContext(ctx, "crosslink getter")
	defer crosslinkGetterSpan.Finish()

	queryParams := table.NewQueryParameters(
		table.ValueParam("$transport_type", ydb.UTF8Value(transportType)),
		table.ValueParam("$source_to_key", ydb.UTF8Value(sourceToKey)),
		table.ValueParam("$source_from_key", ydb.UTF8Value(sourceFromKey)),
		table.ValueParam("$graph_version", ydb.Int64Value(graphVersion)))
	var res *table.Result
	err := repo.retryer.Do(ctx, table.OperationFunc(func(ctx context.Context, s *table.Session) (err error) {
		statement, err := s.Prepare(ctx, Query)
		if err != nil {
			return err
		}
		_, res, err = statement.Execute(ctx, repo.readTx, queryParams)
		return err
	}),
	)
	if err != nil {
		return nil, err
	}

	crosslinks := make([]Crosslink, res.RowCount())
	for res.NextSet() {
		i := 0
		for res.NextRow() {
			crosslinks[i] = Crosslink{
				FromKey:               TryString(res, "fromKey"),
				ToKey:                 TryString(res, "toKey"),
				FromSlug:              TryString(res, "fromSlug"),
				FromTitleRuGenitive:   TryString(res, "fromTitleRuGenitive"),
				FromTitleRuAccusative: TryString(res, "fromTitleRuAccusative"),
				FromTitleRuNominative: strings.Split(TryString(res, "from"), ",")[0],
				ToSlug:                TryString(res, "toSlug"),
				ToTitleRuGenitive:     TryString(res, "toTitleRuGenitive"),
				ToTitleRuAccusative:   TryString(res, "toTitleRuAccusative"),
				ToTitleRuNominative:   strings.Split(TryString(res, "to"), ",")[0],
			}
			i += 1
		}
	}
	return crosslinks, nil
}

func (repo *CrosslinksRepository) fillSessionPool() error {
	sessionPoolPreparingContext := context.Background()
	createSessionsAttempts := repo.preparedSessionsCount * 3

	for i := 0; i < repo.preparedSessionsCount && createSessionsAttempts > 0; {
		if session, err := repo.sessionPool.Create(sessionPoolPreparingContext); err == nil {
			if _, err = session.Prepare(sessionPoolPreparingContext, Query); err != nil {
				createSessionsAttempts--
				continue
			}
			if err = repo.sessionPool.Put(sessionPoolPreparingContext, session); err != nil {
				createSessionsAttempts--
				continue
			}
			i++
		} else {
			createSessionsAttempts--
		}
	}
	if createSessionsAttempts == 0 {
		return fmt.Errorf("couldn't fill YDB sessions pool")
	}
	return nil
}

func TryString(res *table.Result, name string) string {
	res.SeekItem(name)
	i := string(res.OUTF8())
	return i
}
