from dataclasses import dataclass


@dataclass(eq=True, frozen=True)
class Associate:
    graph_version: int
    source_from_key: str
    source_to_key: str
    from_key: str
    to_key: str
    transport_type: str

    @staticmethod
    def build(source, associate, transport_type, graph_version=0):
        return Associate(
                transport_type=transport_type,
                graph_version=graph_version,
                source_from_key="c" + str(source['from_id']),
                source_to_key="c" + str(source['to_id']),
                from_key="c" + str(associate['from_id']),
                to_key="c" + str(associate['to_id'])
            )


@dataclass(eq=True, frozen=True)
class Point:
    key: str
    slug: str
    title: str
    title_ru_accusative: str
    title_ru_genitive: str

    @staticmethod
    def build_from_point(associate):
        return Point(
                key="c" + str(associate['from_id']),
                slug=associate['from_slug'],
                title=associate['from'],
                title_ru_accusative=(associate['from_title_ru_accusative']
                                     if associate['from_title_ru_accusative'] else ''),
                title_ru_genitive=(associate['from_title_ru_genitive']
                                   if associate['to_title_ru_genitive'] else '')
        )

    @staticmethod
    def build_to_point(associate):
        return Point(
            key="c" + str(associate['to_id']),
            slug=associate['to_slug'],
            title=associate['to'],
            title_ru_accusative=(associate['to_title_ru_accusative']
                                 if associate['to_title_ru_accusative'] else ''),
            title_ru_genitive=(associate['to_title_ru_genitive']
                               if associate['to_title_ru_genitive'] else '')
        )


def build_data_objects(raw_data_iterator, transport_type, graph_version=0):
    points, associates = set(), set()
    for entry in raw_data_iterator:
        source, assocs = entry['source'], entry['assocs']
        for assoc in assocs:
            points.add(Point.build_from_point(assoc))
            points.add(Point.build_to_point(assoc))
            associates.add(Associate.build(source, assoc, transport_type, graph_version))
    return points, associates
