import os
import logging.config

import asyncio
from aiohttp import web

from travel.library.python.rasp_vault.api import get_secret

from travel.rasp.deploy_notifier.notifier import Notifier
from travel.rasp.deploy_notifier.notifier.handler import LoadTestingHandler, StartrekHandler, TelegramHandler
from travel.rasp.deploy_notifier.notifier.producer import HTTPProducer
from travel.rasp.deploy_notifier.notifier.utils.http_listener import HTTPListener

logging.config.dictConfig({
    'version': 1,
    'formatters': {
        'qloud': {
            'class': 'ylog.QloudJsonFormatter',
        }
    },
    'handlers': {
        'stream': {
            'class': 'logging.StreamHandler',
            'level': 'DEBUG',
            'formatter': 'qloud'
        },
        'file': {
            'class': 'logging.FileHandler',
            'filename': '/var/log/app/app.log',
            'level': 'DEBUG',
            'formatter': 'qloud',
        },
    },
    'loggers': {
        'application': {
            'handlers': ['stream', 'file'],
            'level': 'DEBUG'
        }
    },
})

log = logging.getLogger('application')


async def ping(request: web.Request) -> web.Response:
    log.info('ping')
    return web.Response(text='pong!')


def get_notifier():

    events = asyncio.Queue()

    http_producer = HTTPProducer(events)
    telegram_handler = TelegramHandler(
        get_secret('rasp-robot-tokens-and-passwords.deploy-notifier-telegram-token'),
        os.getenv('TELEGRAM_CHAT_ID'),
        ('DEPLOYED',)
    )

    handlers = [
        StartrekHandler(
            get_secret('rasp-robot-tokens-and-passwords.startrek-oauth-token'),
            os.getenv('STARTREK_LOGIN'),
            os.getenv('STARTREK_QUEUES', '').split(','),
            post_statuses=('DEPLOYED',)
        ),
        telegram_handler,
    ]

    if os.getenv('RUN_LOAD_TESTING'):
        handlers.append(LoadTestingHandler(
            get_secret('rasp-robot-tokens-and-passwords.sandbox-oauth-token'),
            os.getenv('SANDBOX_OWNER'),
            'RASP_LOAD_START_FROM_CONFIG',
            additional_params=[
                {
                    "name": "token_owner",
                    "value": os.getenv('SANDBOX_OWNER'),
                }, {
                    "name": "token_name",
                    "value": "rasp-qloud-oauth",
                }, {
                    "name": "config_path",
                    "value": os.getenv('SANDBOX_LOAD_TESTING_CONFIG_PATH'),
                }, {
                    "name": "st_token_name",
                    "value": "STARTREK_TOKEN",
                }, {
                    "name": "run_on_specific_apps",
                    "value": True,
                },
            ]
        ))

    routes = [
        web.post('/source/{name}', http_producer),
        web.get('/ping', ping),
    ]

    additional_coroutines = [
        HTTPListener(host='localhost', port=8000, routes=routes).start,
        telegram_handler.start
    ]

    return Notifier(handlers, events, additional_coroutines)
