import re
from typing import Iterable, Set, Text

import startrek_client

from travel.rasp.deploy_notifier.notifier import Event
from travel.rasp.deploy_notifier.notifier.handler import BaseHandler
from travel.rasp.deploy_notifier.notifier.utils import LoggingBase
from travel.rasp.deploy_notifier.notifier.utils.decorators import ensure_needs_handling


class StartrekHandler(BaseHandler, LoggingBase):
    HTTP_TEMPLATES = {
        'qloud': 'web_qloud_startrek.j2',
        'sandbox': 'web_sandbox_startrek.j2'
    }

    def __init__(self, token: str, login: str, queues: Iterable[str], debug: bool=False,
                 debug_queues: Iterable[str]=None, post_statuses: Iterable[str]=()) -> None:
        super().__init__()
        self._client = startrek_client.Startrek(token=token, useragent='rasp-notifier')
        self._ticket_re = re.compile(r'(?:{})-\d+'.format('|'.join(queues)))
        self._debug = debug
        self._debug_queues = set() if debug_queues is None else set(debug_queues)
        self._post_statuses = set(post_statuses)
        self._login = login

    def _get_tickets_from_changelog(self, changelog: str) -> Set[str]:
        return set(self._ticket_re.findall(changelog))

    async def _is_already_posted(self, issue: startrek_client.collections.Issues) -> bool:
        for comment in issue.comments.get_all():
            if comment.createdBy.login == self._login:
                return True
        return False

    async def _send_startrek_comment(self, issue_id: str, text: Text) -> None:
        try:
            issue = self._client.issues[issue_id]
            already_posted = await self._is_already_posted(issue)
            if not already_posted:
                issue.comments.create(text=text)
        except startrek_client.exceptions.StartrekError:
            self._logger.exception('Error while Startrek call.')

    async def _post_to_tickets(self, issues: Iterable[str], text: Text) -> None:
        for issue_id in issues:
            if not self._debug or issue_id in self._debug_queues:
                await self._send_startrek_comment(issue_id, text)

    async def handle_qloud(self, event: Event) -> None:
        self._logger.info(f"StartrekHandler.handle_qloud event: {event}")
        if event.data.get('status') not in self._post_statuses:
            return
        await self._post_to_tickets(
            self._get_tickets_from_changelog(event.data.get('comment', '')),
            self._jinja_env.get_template('qloud_startrek.j2').render(**event.data),
        )

    @ensure_needs_handling('startrek')
    async def handle_http(self, event: Event) -> None:
        self._logger.info(f"StartrekHandler.handle_http event: {event}")
        if event.data.get('status') not in self._post_statuses:
            return
        await self._post_to_tickets(
            self._get_tickets_from_changelog(event.data.get('comment', '')),
            self._jinja_env.get_template(self.HTTP_TEMPLATES[event.data['_source']]).render(**event.data),
        )
