import asyncio

from typing import Callable, Coroutine, Iterable

from travel.rasp.deploy_notifier.notifier import Event
from travel.rasp.deploy_notifier.notifier.handler import BaseHandler


class Notifier:
    def __init__(self, handlers: Iterable[BaseHandler], events: asyncio.Queue,
                 additional_coroutines: Iterable[Callable[[], Coroutine]]=()) -> None:
        self._events = events
        self._handlers = tuple(handlers)
        self._additional_coroutines = additional_coroutines

    async def _handle_event(self, event: Event) -> None:
        for handler in self._handlers:
            await handler(event)

    async def _handle_events(self) -> None:
        while True:
            event = await self._events.get()
            await self._handle_event(event)

    def start(self) -> None:
        loop = asyncio.get_event_loop()
        coroutines = asyncio.wait([self._handle_events()] +
                                  [coroutine() for coroutine in self._additional_coroutines])
        loop.run_until_complete(coroutines)
