from typing import AsyncGenerator, Dict

import asyncio

from travel.rasp.deploy_notifier.notifier import Event
from travel.rasp.deploy_notifier.notifier.producer import BaseProducer
from travel.rasp.deploy_notifier.notifier.utils import LoggingBase
from travel.rasp.deploy_notifier.notifier.utils.api_clients import QloudAPI


class QloudProducer(BaseProducer, LoggingBase):
    TITLE = 'qloud'

    def __init__(self, project: str, qloud_api: QloudAPI, queue: asyncio.Queue, polling_delay: float=5) -> None:
        super().__init__()
        self._last_state = {}
        self._project = project
        self._qloud_api = qloud_api
        self._polling_delay = polling_delay
        self._queue = queue

    async def _get_qloud_statuses(self) -> Dict[str, str]:
        new_statuses = {}
        async for environment_id, status in self._qloud_api.get_project_environments_statuses(self._project):
            new_statuses[environment_id] = status
        return new_statuses

    async def _get_qloud_updates(self) -> AsyncGenerator[Event, None]:
        while True:
            new_statuses = await self._get_qloud_statuses()
            for environment, status in new_statuses.items():
                if self._last_state.get(environment) == status:
                    continue
                self._logger.info(f'Found a changed environment: {environment}.')

                environment_info = await self._qloud_api.get_environment_info(environment)
                environment_info['url'] = self._qloud_api.get_environment_url(environment)

                event = self._make_event(environment_info)
                self._logger.info(f'QloudProducer new event: {event}')
                yield event
            self._last_state = new_statuses
            await asyncio.sleep(self._polling_delay)

    async def start(self) -> None:
        self._logger.info('Starting new qloud producer.')
        self._last_state = await self._get_qloud_statuses()
        async for update in self._get_qloud_updates():
            await self._queue.put(update)
