import itertools
from typing import Optional

import aiohttp
import asyncio

from travel.rasp.deploy_notifier.notifier.utils import LoggingBase


class APIBase(LoggingBase):
    def __init__(self, token: str, base_url: str, retries_count: Optional[int]=None) -> None:
        self._base_url = base_url
        self._retries_count = retries_count
        self._session = aiohttp.ClientSession(headers={'Authorization': f'OAuth {token}'})

    @staticmethod
    def _get_bad_request_timeout(restart_count: int, backoff_factor: float=5.0, max_sleep_period: float=300) -> float:
        return max(backoff_factor * 2 ** restart_count, max_sleep_period)

    async def _make_query(self, url: str, method: str='get', **kwargs) -> dict:
        request_url = self._base_url + url
        for request_number in itertools.count():
            async with getattr(self._session, method)(request_url, **kwargs) as response:
                try:
                    response.raise_for_status()
                    return await response.json()
                except aiohttp.ClientResponseError:
                    self._logger.exception(f'Error while handling {request_url} request. Sleeping.')
                    if request_number == self._retries_count:
                        raise
                    await asyncio.sleep(self._get_bad_request_timeout(request_number))
