# -*- coding: utf-8 -*-

from collections import defaultdict

from django.utils import translation

from travel.rasp.export.export.v3.core.errors import UserUseError
from travel.rasp.export.export.v3.views.base_view import error_view


AVAILABLE_LANGUAGES = ('ru', 'uk', 'tr')

AVAILABLE_NATIONAL_VERSIONS = ('RU', 'TR', 'UA')

DEFAULT_NATIONAL_VERSION_BY_LANG = defaultdict(lambda: 'ru', {
    'ru': 'ru',
    'uk': 'ua',
    'tr': 'tr',
})

NATIONAL_VERSION_TO_TLD = {
    'ru': 'ru',
    'tr': 'com.tr',
    'ua': 'ua',
}

COUNTRY_ISO_TO_NATIONAL_VERSION = {
    'TR': 'tr',
    'RU': 'ru',
    'UA': 'ua',
}


def error_view_on_failure(func):
    def wrapper(*args, **kwargs):
        try:
            func(*args, **kwargs)
        except UserUseError as ex:
            return error_view(args[1], ex)
    return wrapper


class LanguageAndNationalVersion(object):

    @error_view_on_failure
    def process_request(self, request):
        lang_str = request.GET.get('lang', None)
        if not request.path.startswith('/v3'):
            lang = lang_str if lang_str in AVAILABLE_LANGUAGES else 'ru'

            national_version = request.GET.get('national_version', '').lower()
            if national_version.upper() not in AVAILABLE_NATIONAL_VERSIONS:
                national_version = DEFAULT_NATIONAL_VERSION_BY_LANG[lang]

            request.national_version = national_version
        else:
            if not lang_str:
                lang, lang_region = 'ru', 'RU'
            else:
                try:
                    lang, lang_region = lang_str.split('_')
                except ValueError:
                    raise UserUseError(u'Указанный формат параметра lang - {} не поддерживается. '
                                       u'Поддерживаемый формат lang=language_region, '
                                       u'language — код языка в формате ISO 639, '
                                       u'region — код страны в формате ISO 3166. '
                                       u'Пример: ru_RU.'.format(lang_str))

            lang, lang_region = lang.lower(), lang_region.upper()

            if lang not in AVAILABLE_LANGUAGES:
                raise UserUseError(u'Указанный язык {} не поддерживается.'.format(lang))

            try:
                request.national_version = COUNTRY_ISO_TO_NATIONAL_VERSION[lang_region]
            except KeyError:
                raise UserUseError(u'Указанный код страны {} не поддерживается.'.format(lang_region))

        translation.activate(lang)
        request.tld = NATIONAL_VERSION_TO_TLD.get(request.national_version, 'ru')
        request.language_code = translation.get_language()
