# coding: utf8
from __future__ import unicode_literals, absolute_import, division, print_function

import json
import re
from datetime import datetime

from django.conf import settings

from common.dynamic_settings.default import conf
from common.models_abstract.schedule import ExpressType
from common.models.schedule import RTStation
from common.models.transport import TransportType, TransportSubtype
from common.utils import marketstat

from route_search.shortcuts import find


TRAIN_SUBTYPES_CODES = {'lastdal'}

users_search_log = marketstat.DsvLog(settings.USERS_SEARCH_LOG) if settings.USERS_SEARCH_LOG else None


def log_search(log, request, point_from, point_to, when, segments):
    """ Логируем факт поиска в YT """
    try:
        suburban_code = TransportType.objects.get(id=TransportType.SUBURBAN_ID).code
        t_type_counts = {suburban_code: len(segments)}

        user_agent = request.META.get('HTTP_USER_AGENT')
        user_type = None
        if user_agent:
            if user_agent.startswith('yandex_trains_ios'):
                user_type = 'ios_app'
            elif user_agent.startswith('Android app'):  # example: Android app: 3.20(320)
                user_type = user_agent

        # Нас интересуют только поиски из приложений
        if not user_type:
            return

        when_str = when.strftime("%Y-%m-%d") if when else None

        log_data = {
            'from_id': point_from.point_key,
            'to_id': point_to.point_key,
            'transport_type': suburban_code,
            'when': when_str,
            't_type_counts': json.dumps(t_type_counts, separators=(',', ':')),
            'national_version': getattr(request, 'NATIONAL_VERSION', None),
            'service': 'export',
            'user_type': user_type,
            'tskv_format': 'rasp-users-search-log',
            'referer': request.META.get('HTTP_REFERER'),
            'device_uuid': request.GET.get('uuid')
        }

        users_search_log.log(request, log_data)
    except Exception:
        log.exception('Error when writing to user-search-log')


def get_transport_type(thread):
    transport_data = {}
    if thread.express_type == 'aeroexpress':
        transport_data['express_type'] = thread.express_type
    elif thread.express_type == 'express':
        transport_data['express_type'] = thread.express_type

    if thread.t_subtype and thread.t_subtype_id != TransportSubtype.SUBURBAN_ID:
        color = thread.t_subtype.color.color if thread.t_subtype.color else None
        code = thread.t_subtype.code

        if code in TransportSubtype.get_train_search_codes():
            title = thread.t_subtype.L_title_suburban()
            if title and code in TRAIN_SUBTYPES_CODES:
                title = '{}{}'.format(title, conf.EXPORT_TRAIN_SUBTYPE_POSTFIX)

            transport_data['express_type'] = ExpressType.EXPRESS
        else:
            title = thread.t_subtype.L_title()

        transport_data['subtype'] = {
            'code': code,
            'title': title,
            'color': color
        }
    return transport_data


def get_days_and_except_texts(today, thread, shift, next_plan=None):
    days_data = thread.L_days_text_dict(
        shift=shift,
        thread_start_date=today,
        next_plan=next_plan,
        show_days=True)
    return days_data['days_text'], days_data.get('except_days_text')


def clean_number(thread):
    if thread.t_type.id == TransportType.TRAIN_ID:
        return thread.number
    return re.sub(ur'[^0-9/]+', u'', thread.number, flags=re.U)


def find_suburban(point_from, point_to, local_date_from):
    return find(point_from, point_to, local_date_from, TransportType.objects.get(code='suburban'))


def find_suburban_and_train(point_from, point_to, local_date_from):
    return find(
        point_from=point_from,
        point_to=point_to,
        date_from=local_date_from,
        t_type=[TransportType.objects.get(id=TransportType.SUBURBAN_ID)],
        add_train_subtypes=TransportSubtype.get_train_search_codes()
    )


def format_dt(dt, time_format):
    return dt.strftime(time_format) if time_format else dt.isoformat()


def cut_excess_str(string):
    if isinstance(string, basestring):
        return string[:1000] or None


def get_thread_type(thread):
    for thread_type in ('assignment', 'cancel', 'update'):
        if getattr(thread, thread_type):
            return thread_type


def set_key(obj, key, value, skip_empty=True):
    if not skip_empty or value == 0 or value:
        obj[key] = value


def get_facilities_list(facilities):
    if not facilities:
        return []

    facilities_list = []
    for facility in facilities:
        facility_dict = {
            'code': facility.code or None,
            'title': facility.L_title() or None,
            'icon': facility.icon.name and facility.icon.url or None
        }
        facilities_list.append(facility_dict)
    return facilities_list


def get_first_rtstations_by_threads(threads):
    first_rtstations = (
        RTStation.objects.
        filter(tz_arrival__isnull=True, thread__in=list(threads)).
        select_related('station', 'thread')
    )

    first_rtstations_by_threads = {rts.thread: rts for rts in first_rtstations}
    return first_rtstations_by_threads


def fill_thread_local_start_dt(objs_with_thread):
    threads = {obj.thread for obj in objs_with_thread}
    first_rtstations_by_threads = get_first_rtstations_by_threads(threads)

    for obj in objs_with_thread:
        first_rtstation = first_rtstations_by_threads[obj.thread]
        naive_start_dt = datetime.combine(obj.start_date, obj.thread.tz_start_time)
        obj.thread_local_start_dt = first_rtstation.get_departure_loc_dt(naive_start_dt)
