# coding: utf8
from __future__ import unicode_literals, absolute_import, division, print_function

import logging
from functools import partial
from itertools import chain
from multiprocessing import TimeoutError

import pytz
from django.conf import settings

from common.apps.suburban_events.api import get_states_by_schedule_routes
from common.data_api.platforms.helpers import ScheduleRoutePlatformsBatch
from common.data_api.platforms.instance import platforms as platforms_client
from common.models.schedule import RThread, TrainSchedulePlan
from common.models.transport import TransportSubtype
from common.models_utils import fetch_related
from common.models_utils.i18n import RouteLTitle
from travel.rasp.library.python.common23.date import environment
from common.utils.date import MSK_TZ
from travel.rasp.library.python.common23.logging import log_run_time
from common.utils.multiproc import get_pool

from stationschedule import SUBURBAN, get_schedule_class
from stationschedule.facilities import fill_suburban_facilities

from travel.rasp.export.export.v3.core.helpers import (
    get_days_and_except_texts, get_transport_type, clean_number, format_dt, cut_excess_str, get_thread_type, set_key,
    get_facilities_list, fill_thread_local_start_dt
)
from travel.rasp.export.export.v3.core.suburban_events import get_route_state_dict
from travel.rasp.export.export.v3.core.teasers import get_station_schedule_teasers
from travel.rasp.export.export.v3.views.utils import get_server_time, get_day_start_end_utc_in_iso


log = logging.getLogger(__name__)
log_run_time = partial(log_run_time, logger=log, log_level=logging.DEBUG)


def get_station_schedule_on_date(station, esr_code, timezone, request_date, national_version=None, lang=None):
    station_schedule_data, schedule_routes, now_aware, direction_title_by_code, timezone, next_plan, train_routes \
        = get_suburban_train_schedules(station, esr_code, timezone, request_date=request_date,
                                       national_version=national_version, lang=lang)
    schedule_routes += train_routes
    fill_suburban_facilities(schedule_routes)
    fill_thread_local_start_dt(schedule_routes)

    if settings.ENABLE_SUBURBAN_STATES:
        with log_run_time('get states for {} schedule routes'.format(len(schedule_routes))):
            try:
                schedule_states = get_states_by_schedule_routes(schedule_routes, all_keys=True)
            except Exception as ex:
                log.exception(repr(ex))
                schedule_states = {}
    else:
        schedule_states = {}

    dynamic_platforms = ScheduleRoutePlatformsBatch()
    dynamic_platforms.try_load(platforms_client, schedule_routes)

    station_schedule = [build_thread_data_on_date(s, now_aware, direction_title_by_code, timezone, next_plan,
                                                  route_state=schedule_states.get(s),
                                                  dynamic_platforms=dynamic_platforms)
                        for s in schedule_routes]

    station_schedule_data['threads'] = station_schedule
    day_start_utc_iso, day_end_utc_iso = get_day_start_end_utc_in_iso(request_date, station.pytz)
    station_schedule_data['date_time'].update({'day_start_utc': day_start_utc_iso,
                                               'day_end_utc': day_end_utc_iso,
                                               'date': request_date.strftime("%Y-%m-%d")})
    return station_schedule_data


def get_suburban_train_schedules(station, esr_code, timezone, request_date=None, national_version=None, lang=None):
    with get_pool(pool_size=1) as pool:
        train_search = pool.apply_async(common_train_station_schedule, (station, request_date))
        station_schedule_data, schedule_routes, now_aware, direction_title_by_code, timezone, next_plan = \
            common_station_schedule(station, esr_code, timezone, request_date=request_date,
                                    national_version=national_version, lang=lang)
        try:
            train_routes = train_search.get(timeout=1)
        except TimeoutError:
            train_routes = []

    if train_routes:
        plans_qs = TrainSchedulePlan.objects.filter(id__in=[sr.schedule_item.schedule_plan_id
                                                            for sr in chain(schedule_routes, train_routes)])
        current_plan, next_plan = TrainSchedulePlan.get_current_and_next(environment.today(), qs=plans_qs)

    return station_schedule_data, schedule_routes, now_aware, direction_title_by_code, timezone, next_plan, train_routes


def get_station_schedule_on_all_days(station, esr_code, timezone, national_version=None, lang=None):
    time_format = '%H:%M'
    station_schedule_data, schedule_routes, now_aware, direction_title_by_code, timezone, next_plan, train_routes \
        = get_suburban_train_schedules(station, esr_code, timezone, national_version=national_version, lang=lang)

    schedule_routes += train_routes
    station_schedule = [build_thread_data(s, now_aware, direction_title_by_code, timezone, next_plan, time_format)
                        for s in schedule_routes]
    station_schedule_data['threads'] = station_schedule

    return station_schedule_data


def common_train_station_schedule(station, request_date=None):
    schedule_cls = get_schedule_class(station, t_type_code='train')
    schedule = schedule_cls(station, schedule_date=request_date, event=None)
    schedule.build(schedule_date=request_date)

    schedule_routes = schedule.schedule
    fetch_related([s.thread for s in schedule_routes], 't_subtype', model=RThread)
    schedule_routes = [schedule_route for schedule_route in schedule_routes if not schedule_route.cancel and
                       schedule_route.thread.t_subtype and schedule_route.thread.t_subtype.code in TransportSubtype.get_train_search_codes()]

    RouteLTitle.fetch([schedule_route.thread.L_title for schedule_route in schedule_routes])

    return schedule_routes


def common_station_schedule(station, esr_code, timezone, request_date=None, national_version=None, lang=None):
    now_aware = environment.now_aware()

    schedule_result = {
        'esr': esr_code,
        'date_time': {'server_time': get_server_time()},
    }

    # Формируем расписание
    schedule_cls = get_schedule_class(station, schedule_type=SUBURBAN, t_type_code=SUBURBAN)
    schedule = schedule_cls(station, requested_direction='all', schedule_date=request_date)

    msk_today = now_aware.astimezone(MSK_TZ).date()
    current_plan, next_plan = schedule.current_next_plans(msk_today)

    schedule.build(schedule_date=request_date)

    # Пропускаем нитки отмены
    schedule_routes = [schedule_route for schedule_route in schedule.schedule
                       if not schedule_route.cancel]

    # заполняем направления
    schedule.fill_directions(schedule_routes)

    direction_title_by_code = schedule.direction_title_by_code

    RouteLTitle.fetch([schedule_route.thread.L_title for schedule_route in schedule_routes])

    set_key(schedule_result, 'teasers', get_station_schedule_teasers(station, schedule_routes,
                                                                     national_version=national_version, lang=lang))

    return schedule_result, schedule_routes, now_aware, direction_title_by_code, timezone, next_plan


def build_thread_data(schedule_route, now_aware, direction_title_by_code, timezone, next_plan,
                      time_format=None, dynamic_platforms=None):
    thread = schedule_route.thread
    thread_data = {
        'uid': thread.uid,
        'canonical_uid': thread.canonical_uid,
        'number': cut_excess_str(clean_number(thread)),
        'title': thread.L_title(),
        'is_combined': thread.is_combined,
        'stops': cut_excess_str(schedule_route.schedule.L_stops())
    }

    thread_tz_today = now_aware.astimezone(schedule_route.thread.pytz).date()

    shift = schedule_route.rtstation.calc_days_shift(event=schedule_route.event, event_tz=timezone,
                                                     start_date=schedule_route.start_date)
    days_text, except_text = get_days_and_except_texts(thread_tz_today, schedule_route.thread, shift, next_plan)

    if schedule_route.is_last_station:
        thread_data['terminal'] = 'terminal'

    if schedule_route.loc_arrival_datetime:
        arrival = (
            schedule_route.loc_arrival_datetime.astimezone(timezone) if timezone
            else schedule_route.loc_arrival_datetime
        )
        thread_data['arrival'] = {'time': format_dt(arrival, time_format)}

    if schedule_route.loc_departure_datetime:
        departure = (
            schedule_route.loc_departure_datetime.astimezone(timezone) if timezone
            else schedule_route.loc_departure_datetime
        )
        thread_data['departure'] = {'time': format_dt(departure, time_format)}
        set_key(thread_data['departure'], 'platform', ScheduleRoutePlatformsBatch.get_departure_safe(
            dynamic_platforms, schedule_route, cut_excess_str(schedule_route.rtstation.L_platform())
        ))

    if hasattr(schedule_route.schedule_item, 'direction_code'):
        direction_title = direction_title_by_code.get(schedule_route.schedule_item.direction_code)
    else:
        direction_title = None

    set_key(thread_data, 'direction', cut_excess_str(direction_title))
    set_key(thread_data, 'days', cut_excess_str(days_text))
    set_key(thread_data, 'except', cut_excess_str(except_text))
    set_key(thread_data, 'type', get_thread_type(thread))
    set_key(thread_data, 'transport', get_transport_type(thread))
    return thread_data


def build_thread_data_on_date(schedule_route, now_aware, direction_title_by_code, timezone, next_plan,
                              route_state=None, dynamic_platforms=None):
    thread_data = build_thread_data(schedule_route, now_aware, direction_title_by_code, timezone, next_plan,
                                    dynamic_platforms=dynamic_platforms)
    set_key(thread_data, 'facilities', get_facilities_list(schedule_route.suburban_facilities))
    set_key(thread_data, 'start_time', schedule_route.thread_local_start_dt.isoformat())
    if schedule_route.loc_departure_datetime:
        thread_data['departure']['time_utc'] = schedule_route.loc_departure_datetime.astimezone(pytz.utc).isoformat()

    if schedule_route.loc_arrival_datetime:
        thread_data['arrival']['time_utc'] = schedule_route.loc_arrival_datetime.astimezone(pytz.utc).isoformat()

    if route_state:
        state_dict = get_route_state_dict(route_state)
        for state_type, state in state_dict.items():
            if state_type == 'thread':
                thread_data['thread_state'] = state
            else:
                thread_data[state_type]['state'] = state
    return thread_data
