# coding: utf8
from __future__ import unicode_literals, absolute_import, division, print_function

from itertools import chain

import pytz

from common.apps.suburban_events.utils import EventStateType


TIME_FORMAT = '%Y-%m-%d %H:%M'

STATE_TYPES = ('arrival', 'departure')  # Список типов состояний, которые должен отдавать экспорт.
SEARCH_STATE_TYPES = ('from_arrival', 'to_departure')  # Список дополнительных типов для поиска.

STATE_NAME_BY_TYPE = {
    SEARCH_STATE_TYPES[0]: ('departure', 'arrival_state'),
    SEARCH_STATE_TYPES[1]: ('arrival', 'departure_state'),
    STATE_TYPES[0]: ('arrival', 'state'),
    STATE_TYPES[1]: ('departure', 'state')
}
DYNAMIC_DATA_STATE_NAME_BY_TYPE = {
    SEARCH_STATE_TYPES[0]: 'arrival_state',
    SEARCH_STATE_TYPES[1]: 'departure_state',
}


def get_event_state_dict(state):
    event_state_dict = {}
    if state.type:
        event_state_dict['type'] = state.type

    if state.type == EventStateType.FACT:
        dt = pytz.timezone(state.tz).localize(state.dt)
        event_state_dict['fact_time'] = dt.isoformat()

    if state.type in [EventStateType.POSSIBLE_DELAY, EventStateType.FACT_INTERPOLATED, EventStateType.FACT]:
        event_state_dict['minutes_from'] = state.minutes_from
        event_state_dict['minutes_to'] = state.minutes_to

    return event_state_dict


def get_segment_state_dict(segment_state):
    segment_state_dict = {}

    for state_type in chain(STATE_TYPES, SEARCH_STATE_TYPES):
        state = getattr(segment_state, state_type)
        if not state:
            continue

        event_state_dict = get_event_state_dict(state)
        segment_state_dict[state_type] = event_state_dict

        if state.key:
            event_state_dict['key'] = state.key

    return segment_state_dict


def get_route_state_dict(route_state):
    route_state_dict = {}

    for state_type in STATE_TYPES:
        state = getattr(route_state, state_type)
        if not state:
            continue

        event_state_dict = get_event_state_dict(state)
        route_state_dict[state_type] = event_state_dict

        if state.key:
            event_state_dict['key'] = state.key

    return route_state_dict


def create_station_state_dict(station_state):
    state_dict = {'key': station_state.key}
    for event in ['arrival', 'departure']:
        event_state = getattr(station_state, event)
        if not event_state:
            continue

        state_dict[event] = get_event_state_dict(event_state)

    return state_dict


def create_thread_state_dict(thread_state):
    state_dict = get_event_state_dict(thread_state)
    state_dict['key'] = thread_state.key

    return state_dict
