# coding: utf8
from __future__ import unicode_literals, absolute_import, division, print_function

import logging
from datetime import datetime
from functools import partial

from django.conf import settings
from rest_framework import status

from common.apps.facility.shortcuts import get_suburban_thread_facilities
from common.apps.suburban_events.api import get_states_for_thread
from common.data_api.platforms.helpers import PathPlatformsBatch
from common.data_api.platforms.instance import platforms as platforms_client
from common.models.schedule import TrainSchedulePlan, StationSchedule
from common.models_utils.i18n import RouteLTitle
from travel.rasp.library.python.common23.date import environment
from common.utils.date import MSK_TZ
from travel.rasp.library.python.common23.logging import log_run_time

from travel.rasp.export.export.v3.core.helpers import (
    get_days_and_except_texts, get_transport_type, clean_number, format_dt, cut_excess_str, get_thread_type, set_key,
    get_facilities_list)
from travel.rasp.export.export.v3.core.errors import UserUseError
from travel.rasp.export.export.v3.core.suburban_events import create_thread_state_dict, create_station_state_dict, STATE_TYPES
from travel.rasp.export.export.v3.core.tariffs.tariffs import get_thread_suburban_tariffs
from travel.rasp.export.export.v3.core.teasers import get_thread_teasers
from travel.rasp.export.export.v3.views.utils import append_station_title_prefix
from travel.rasp.export.export.v3.views.utils import dt_to_tz_date, get_today_tomorrow_date

log = logging.getLogger(__name__)
log_run_time = partial(log_run_time, logger=log, log_level=logging.DEBUG)


def get_thread_on_date(thread, request_date, national_version=None, lang=None,
                       station_from_id=None, station_to_id=None, disable_cancels=True):
    rtstations = list(thread.path.select_related('station'))
    first_rtstation = rtstations[0]
    first_station_tz = first_rtstation.station.pytz
    departure_date = get_today_tomorrow_date(request_date, first_station_tz)
    if not departure_date:
        departure_date = dt_to_tz_date(request_date, first_station_tz)
    start_date = first_rtstation.calc_thread_start_date(event='departure', event_date=departure_date)
    naive_start_dt = datetime.combine(start_date, thread.tz_start_time)

    if not thread.runs_at(naive_start_dt):
        raise UserUseError(u'Рейс {} не ходит {}'.format(thread.uid, departure_date), status.HTTP_404_NOT_FOUND)

    thread_data = base_thread(thread, naive_start_dt, first_rtstation,
                              national_version=national_version, lang=lang)

    if settings.ENABLE_SUBURBAN_STATES:
        with log_run_time('get states for thread {} and {} rtstations'.format(thread.id, len(rtstations))):
            try:
                thread_state, station_states = get_states_for_thread(
                    thread, start_date, rtstations, all_keys=True, cancels_as_possible_delay=disable_cancels
                )
            except Exception as ex:
                log.exception(repr(ex))
                thread_state, station_states = None, {}
    else:
        thread_state, station_states = None, {}

    if thread_state and 'thread' in STATE_TYPES:
        thread_data['state'] = create_thread_state_dict(thread_state)

    aware_start_dt = thread.pytz.localize(naive_start_dt)
    thread_data['stations'] = build_thread_rtstations(
        rtstations, naive_start_dt, aware_start_dt, station_states=station_states, train_number=thread.number
    )

    facilities = get_suburban_thread_facilities(thread.id, start_date)
    if facilities:
        thread_data['facilities'] = get_facilities_list(facilities)

    add_thread_tariffs(thread, station_from_id, station_to_id, thread_data)

    return thread_data


def get_thread_on_all_days(thread, national_version=None, lang=None, station_from_id=None, station_to_id=None):
    time_format = '%H:%M'
    msk_today = environment.today()
    thread_start_date = thread.first_run(msk_today)
    naive_start_dt = datetime.combine(thread_start_date, thread.tz_start_time)
    rtstations = list(thread.path.select_related('station'))
    first_rtstation = rtstations[0]

    thread_data = base_thread(thread, naive_start_dt, first_rtstation, time_format,
                              national_version=national_version, lang=lang)

    aware_start_dt = thread.pytz.localize(naive_start_dt)
    thread_data['stations'] = build_thread_rtstations(rtstations, naive_start_dt, aware_start_dt, time_format)

    add_thread_tariffs(thread, station_from_id, station_to_id, thread_data)

    return thread_data


def base_thread(thread, naive_start_dt, first_rtstation, time_format=None, national_version=None, lang=None):
    first_station = first_rtstation.station
    RouteLTitle.fetch([thread.L_title])
    thread_start_date = thread.pytz.localize(naive_start_dt).date()
    aware_start_dt = thread.pytz.localize(naive_start_dt)
    msk_departure = aware_start_dt.astimezone(MSK_TZ)
    local_departure = aware_start_dt.astimezone(first_station.pytz)

    thread_data = {
        'canonical_uid': thread.canonical_uid,
        'title': cut_excess_str(thread.L_title()),
        'number': cut_excess_str(clean_number(thread)),
        'is_combined': thread.is_combined,
        'start_time': format_dt(local_departure, time_format),
        'start_time_msk': format_dt(msk_departure, time_format),
    }

    shift = first_rtstation.calc_days_shift(event='departure', start_date=thread_start_date)
    current_plan, next_plan = TrainSchedulePlan.add_to_threads([thread], environment.today())
    days_text, except_text = get_days_and_except_texts(thread_start_date, thread,
                                                       shift, next_plan=next_plan)
    try:
        schedule = StationSchedule.get(thread=thread, station=first_station, event='departure')
        stops_text = schedule.L_stops()
    except StationSchedule.DoesNotExist:
        stops_text = u''
    thread_data['stops'] = cut_excess_str(stops_text)

    set_key(thread_data, 'days', cut_excess_str(days_text))
    set_key(thread_data, 'except', cut_excess_str(except_text))
    set_key(thread_data, 'teasers', get_thread_teasers(thread, national_version=national_version, lang=lang))
    set_key(thread_data, 'type', get_thread_type(thread))
    set_key(thread_data, 'transport', get_transport_type(thread))
    return thread_data


def add_thread_tariffs(thread, station_from_id, station_to_id, thread_data):
    if station_from_id and station_to_id:
        tariffs, tariff = get_thread_suburban_tariffs(thread, station_from_id, station_to_id)
        thread_data['tariffs'], thread_data['tariff'] = tariffs, tariff


def build_thread_rtstations(rtstations, naive_start_dt, aware_start_dt,
                            time_format=None, station_states=None, train_number=None):
    rtstations_list = []
    dynamic_platforms = PathPlatformsBatch(naive_start_dt, train_number)
    dynamic_platforms.try_load(platforms_client, list(rtstations))
    for rtstation in rtstations:
        if rtstation.is_technical_stop:
            continue

        station = rtstation.station
        title = station.L_title()
        popular_title = station.L_popular_title(fallback=False)
        title = append_station_title_prefix(station, title)

        rtstation_data = {
            'title': title or None,
            'is_combined': rtstation.is_combined,
        }
        set_key(rtstation_data, 'popular_title', popular_title)

        esr_code = rtstation.station.get_code('esr')
        if esr_code:
            rtstation_data['esr'] = esr_code

        arr_dep_time = {}
        for event in ['arrival', 'departure']:
            t = getattr(rtstation, 'tz_{}'.format(event))
            if t is not None:
                local_dt = rtstation.get_event_loc_dt(event, naive_start_dt)
                arr_dep_time[event] = int((local_dt - aware_start_dt).total_seconds() / 60)
                arr_dep_time[event + '_local'] = format_dt(local_dt, time_format)

        no_stop = arr_dep_time.get('departure') == arr_dep_time.get('arrival')

        if no_stop:
            rtstation_data['no_stop'] = no_stop
        else:
            rtstation_data.update(arr_dep_time)
            set_key(rtstation_data, 'platform', dynamic_platforms.get_platform(rtstation, rtstation.L_platform()))
            if station_states:
                rts_state = station_states.get(rtstation)
                if rts_state:
                    rtstation_data['state'] = create_station_state_dict(rts_state)

        rtstations_list.append(rtstation_data)

    return rtstations_list
