# coding: utf8

from collections import defaultdict

from common.models.geo import Station, Settlement, Region, StationCode, SuburbanZone
from common.models.transport import TransportType

from travel.rasp.export.export.v3.core.helpers import set_key
from travel.rasp.export.export.v3.views.utils import append_station_title_prefix


CACHE_TIMEOUT = 2 * 7 * 24 * 3600  # 2 weeks


def get_zone(zone):
    markers = zone.markers()
    stations = Station.objects.filter(id__in=[marker.station_id for marker in markers])

    stations_ids_by_title = defaultdict(lambda: defaultdict(list))
    regions_ids = {station.region.id for station in stations if station.region}

    region_stations = Station.objects.filter(region_id__in=regions_ids,
                                             t_type_id__in=[TransportType.TRAIN_ID, TransportType.SUBURBAN_ID])
    for station in region_stations:
        stations_ids_by_title[station.region][station.title].append(station.id)

    suburban_zone = {
        'id': zone.id,
        'zone_stations': export_stations_list(stations, markers,
                                              city=zone.settlement,
                                              stations_ids_by_title=stations_ids_by_title)
    }
    return suburban_zone


def get_all_zones(use_directions=False):
    suburban_zones = []

    for zone in SuburbanZone.objects.all().select_related('settlement'):
        zone_data = {
            'id': zone.id,
            'title': zone.L_title() or None,
            'code': zone.code or None,
            'country': zone.settlement.country.L_title() if zone.settlement.country else None,
            'country_code': zone.settlement.country.code if zone.settlement.country else None,
            'settlement_id': zone.settlement.id,
            'settlement_title': zone.settlement.L_title() or None,
            'settlement_title_genitive': zone.settlement.L_title(case='genitive', fallback=False) or None
        }

        zone_settlements = []

        if use_directions:
            zone_data['majority'] = zone.majority
            stations = Station.objects.filter(
                externaldirectionmarker__external_direction__suburban_zone=zone,
                settlement__isnull=False,
                settlement__hidden=False
            )
            settlements = {station.settlement for station in stations}
        else:
            settlements = zone.settlement_set.filter(hidden=False)

        for settlement in settlements:
            if settlement.latitude is None or settlement.longitude is None:
                continue
            title = settlement.L_title()

            if getattr(settlement, 'use_in_suburban_app_suggests') == 'all_railway_stations':
                title = u'{} {}'.format(title, u'(все вокзалы)')

            settlement_data = {
                'id': settlement.id,
                'title': title,
                'title_genitive': settlement.L_title(case='genitive', fallback=False) or None,
                'longitude': settlement.longitude,
                'latitude': settlement.latitude,
                'geo_id': settlement._geo_id,
                'region': settlement.region.L_title() if settlement.region else None,
                'country': settlement.country.L_title() if settlement.country else None,
                'country_code': settlement.country.code if settlement.country else None,
            }

            if use_directions:
                settlement_data['use_in_suburban_app_suggests'] = settlement.use_in_suburban_app_suggests

            zone_settlements.append(settlement_data)
        set_key(zone_data, 'settlements', zone_settlements)
        suburban_zones.append(zone_data)

    return {'suburban_zones': suburban_zones}


def export_stations_list(stations, markers=None, city=None, stations_ids_by_title=None):
    id_to_code = StationCode.id_to_code_dict_for('esr', stations)
    visited_stations = {}

    regions = Region.objects.in_bulk(station.region_id for station in stations)
    settlements = Settlement.objects.in_bulk(station.settlement_id for station in stations)

    zone_stations = []
    for station in stations:
        try:
            esr_code = id_to_code[station.id]
        except KeyError:
            continue

        if esr_code in visited_stations:
            continue

        title = station.L_title()
        title_genitive = station.L_title(case='genitive', fallback=False)

        popular_title = station.L_popular_title(fallback=False)
        popular_title_genitive = station.L_popular_title(case='genitive', fallback=False)

        title = append_station_title_prefix(station, title)

        direction = get_station_direction(station, markers, city)

        station_data = {
            'esr': esr_code,
            'title': title or None,
            'title_genitive': title_genitive or None,
            'popular_title': popular_title or None,
            'popular_title_genitive': popular_title_genitive or None,
            'station_type_id': station.station_type.id,
            'importance': station.majority_id,
            'region': regions[station.region_id].L_title() if station.region_id in regions else None,
            'direction': direction.L_full_title() if direction else None,
            'direction_id': direction.id if direction else None,
            'yandex_rasp_code': station.id,
            'country': station.country.L_title() if station.country else None,
            'country_code': station.country.code if station.country else None,
        }

        if station.settlement_id in settlements:
            station_data['settlement_title'] = settlements[station.settlement_id].L_title()

        if station.latitude:
            station_data['latitude'] = station.latitude
            station_data['longitude'] = station.longitude

        if station.L_short_title():
            station_data['short_title'] = station.L_short_title()

        if stations_ids_by_title:
            station_data['need_direction'] = (len(stations_ids_by_title.get(station.region, {}).get(station.title, [])) > 1)

        visited_stations[esr_code] = True
        zone_stations.append(station_data)
    return zone_stations


def get_station_direction(station, markers, city):
    if markers:
        for marker in markers:
            if marker.station_id == station.pk:
                return marker.external_direction

    else:
        try:
            return station.externaldirectionmarker_set.filter(
                external_direction__suburban_zone__settlement=city
            ).select_related('external_direction')[0].external_direction

        except IndexError:
            pass

    return None
