# coding: utf8
from __future__ import unicode_literals, absolute_import, division, print_function

import json
from collections import defaultdict

from common.db.mds.clients import mds_s3_common_client
from travel.rasp.export.export.v3.partners.suburban_states import StationThreadObject, _generate_threads_states
from travel.rasp.export.export.v3.views.base_view import BaseView


class NearestSuburbansView(BaseView):
    """
    В tasks генерируются данные о ближайших электричках для ПП.
    Данные складываются в mds.
    При запросе в export от ПП данные достаются из mds
    и к ним добавляются данные об опозданиях.
    """
    def handle(self, request, *args, **kwargs):
        raw_data = json.load(mds_s3_common_client.get_data(key='suburban/nearest_suburban_to_from_center.json'))
        return _get_nearest_suburbans(raw_data)


class NearestSuburbansAllView(BaseView):
    """
    https://st.yandex-team.ru/RASPEXPORT-273
    Аналогично NearestSuburbansView, но для файла со всеми станциями.
    """
    def handle(self, request, *args, **kwargs):
        raw_data = json.load(mds_s3_common_client.get_data(key='suburban/nearest_suburban_to_from_center_all.json'))
        return _get_nearest_suburbans(raw_data)


class NearestSuburbansMainStationsView(BaseView):
    """
    https://st.yandex-team.ru/RASPFRONT-6481
    Аналогично NearestSuburbansView, но для файла с главными станциями центров пригородных зон
    """
    def handle(self, request, *args, **kwargs):
        key = 'suburban/nearest_suburban_main_stations_departures.json'
        raw_data = json.load(mds_s3_common_client.get_data(key=key))
        return _get_main_station_departures(raw_data)


def _get_nearest_suburbans(data):
    uids = set()
    thread_station_objs = []

    for cur_station, nearests in data.items():
        for mov_dir in ['fc', 'tc']:
            for suburban in nearests[mov_dir]:
                uids.add(suburban['uid'])
                suburban['mov_dir'] = mov_dir
                suburban['cur_station'] = cur_station
                thread_station_objs.append(StationThreadObject(suburban))

    result_data = defaultdict(lambda: defaultdict(list))
    for obj, obj_data in _generate_threads_states(thread_station_objs, uids):
        result_data[obj.cur_station][obj.mov_dir].append(obj_data)

    return result_data


def _get_main_station_departures(data):
    uids = set()
    thread_station_objs = []

    for cur_station, nearests in data.items():
        for suburban in nearests:
            uids.add(suburban['uid'])
            suburban['cur_station'] = cur_station
            thread_station_objs.append(StationThreadObject(suburban))

    result_data = defaultdict(lambda: [])
    for obj, obj_data in _generate_threads_states(thread_station_objs, uids):
        result_data[obj.cur_station].append(obj_data)

    return result_data
