# coding: utf8
from __future__ import unicode_literals, absolute_import, division, print_function

import logging
import re
from collections import defaultdict

from django.conf import settings
from iso8601 import parse_date
from mongoengine.context_managers import switch_db

from common.apps.suburban_events.api import gen_station_states_by_keys, SegmentStates, SegmentEventState, get_tss_by_key
from common.apps.suburban_events.models import ThreadStationState
from common.apps.suburban_events.utils import EventStateType, ThreadStationKey, get_threads_suburban_keys
from common.models.schedule import RThread


log = logging.getLogger(__name__)


class StationThreadObject(object):
    def __init__(self, data):
        self.data = data
        self.mov_dir = data.get('mov_dir')
        self.thread_uid = data.get('uid')
        self.station = str(data.get('from_id'))
        self.arrival = data.get('from_arrival', None)
        self.departure = data.get('from_departure', None)
        self.tz_thread_start_date_str = data.get('tz_thread_start_date')
        self.tz_thread_start_date = parse_date(data.get('tz_thread_start_date'), default_timezone=None)

    def __eq__(self, other):
        return self.compare_str == other.compare_str

    def __hash__(self):
        return self.get_hash

    def __getattr__(self, attr_name):
        return self.data.get(attr_name)

    @property
    def get_hash(self):
        if self._has is None:
            self._hash = hash(self.compare_str)

        return self._hash

    @property
    def compare_str(self):
        if self._compare_str is None:
            self._compare_str = '{thread_uid}___{thread_start_date}___{station}___{arrival}'.format(
                thread_uid=self.thread_uid,
                thread_start_date=self.tz_thread_start_date_str,
                station=self.station,
                arrival=self.arrival
            )

        return self._compare_str


def _generate_threads_states(thread_station_objs, uids):
    threads = list(RThread.objects.filter(uid__in=uids))
    threads_data = {thread.uid: thread for thread in threads}

    try:
        segments_states = _get_states_for_obj(thread_station_objs, threads_data)
    except Exception as ex:
        log.exception('Не удалось получить информацию об опозданиях. {}'.format(repr(ex)))
        segments_states = {}

    for obj in thread_station_objs:
        obj_data = {
            'transport_subtype': obj.transport_subtype,
            'touch_url': obj.touch_url,
            'express': obj.express,
            'travel_time': obj.travel_time,
            'time': obj.time,
            'name': obj.name,
            'uid': obj.uid
        }

        states = segments_states.get(obj)
        if states:
            states_dict = {}
            for event in ['arrival', 'departure']:
                event_state = getattr(states, event, None)
                if event_state and event_state.type == EventStateType.POSSIBLE_DELAY:
                    states_dict[event] = {
                        'type': 'possible_delay',
                        'title': {
                            'ru': 'возможно опоздание',
                            'uk': 'можливе запiзення'
                        }
                    }
            if states_dict:
                obj_data['states'] = states_dict

        yield obj, obj_data


def _prepare_title(t):
    t = re.sub(r'\<[^>]*\>', '', t)
    t = t.replace('&nbsp;', ' ')
    t = t[0].upper() + t[1:]
    return t


def _get_states_for_obj(segments, threads_data):
    objs_keys = _get_keys_by_objs(segments, threads_data)
    station_thread_obj_states = _get_obj_states(objs_keys)

    return station_thread_obj_states


def _get_keys_by_objs(thread_station_objects, threads_data):
    suburban_keys = get_threads_suburban_keys([
        threads_data[obj.thread_uid]
        for obj in thread_station_objects
        if obj.thread_uid in threads_data
    ])
    keys = {}

    for obj in thread_station_objects:
        if obj.thread_uid not in threads_data:
            continue

        suburban_key = suburban_keys.get(threads_data[obj.thread_uid].id)
        if not suburban_key:
            continue

        thread_start_date = obj.tz_thread_start_date
        keys[obj] = ThreadStationKey(
            suburban_key, thread_start_date, obj.station,
            arrival=obj.arrival,
            departure=obj.departure
        )

    return keys


def _get_obj_states(keys):
    with switch_db(ThreadStationState, settings.SUBURBAN_EVENTS_DATABASE_NAME + '_no_timeout'):
        station_states_by_keys = gen_station_states_by_keys(keys.values())

    objs_states = defaultdict(SegmentStates)
    for obj, tss_key in keys.items():
        for event in ['arrival', 'departure']:
            state_name = event + '_state'
            station_state = get_tss_by_key(station_states_by_keys, tss_key)
            if station_state:
                event_state = station_state.get(state_name)
                if event_state and obj.thread_uid == event_state['thread_uid']:
                    setattr(objs_states[obj], event, SegmentEventState(
                        key=tss_key.to_str(),
                        type_=event_state['type'],
                        minutes_from=event_state.get('minutes_from'),
                        minutes_to=event_state.get('minutes_to'),
                        tz=station_state['tz'],
                        dt=event_state.get('dt'),
                    ))
    return objs_states
