# coding: utf8
from __future__ import unicode_literals, absolute_import, division, print_function

import json
import logging
import requests
from typing import List, Dict, Optional, Iterable
from collections import namedtuple, defaultdict
from dateutil import parser

from django.conf import settings

from common.models.schedule import Company
from common.models.tariffs import SuburbanSellingFlow, TariffType
from common.models_abstract.schedule import ExpressType
from route_search.models import RThreadSegment
from travel.library.python.tracing.instrumentation import traced_function
from travel.rasp.export.export.v3.selling import suburban_selling_breaker


log = logging.getLogger(__name__)


COMPANIES_WITH_SELLING_TARIFFS = [
    Company.CPPK_ID, Company.AEROEXPRESS_ID, Company.CPPK_AEROEX_ID,
    Company.SZPPK_ID, Company.MTPPK_ID, Company.SZPPK_MTPPK_ID,
    Company.BASHPPK_ID, Company.SAMPPK_BASHPPK_ID, Company.SODRUZHESTVO_ID
]

IM_COMPANIES = [
    Company.SZPPK_ID, Company.MTPPK_ID, Company.SZPPK_MTPPK_ID,
    Company.BASHPPK_ID, Company.SAMPPK_BASHPPK_ID, Company.SODRUZHESTVO_ID
]

SELLING_V1 = '1'
SELLING_V2 = '2'
SELLING_V3 = '3'


TariffKey = namedtuple('TariffKey', ['date', 'station_from', 'station_to', 'tariff_type', 'company'])
TariffUndatedKey = namedtuple('TariffUndatedKey', ['station_from', 'station_to', 'tariff_type', 'company'])


def get_tariff_key(segment):
    # type: (RThreadSegment) -> Optional[TariffKey]

    if segment.thread.company.id not in COMPANIES_WITH_SELLING_TARIFFS:
        return

    # Тариф необходим для матчинга цен.
    # Тип экспресса и подобные параметры в общем случае не позволяют однозначно определить тариф
    # см. кейс (3) https://st.yandex-team.ru/RASPFRONT-8735#5fb61b43c68ee01b7c08cd53
    if not hasattr(segment, 'base_tariff'):
        return

    return TariffKey(
        date=segment.departure.date().strftime('%Y-%m-%d'),
        station_from=segment.station_from.id,
        station_to=segment.station_to.id,
        tariff_type=segment.base_tariff.type.code,
        company=segment.thread.company.id
    )


def _get_undated_key(tariff_key):
    return TariffUndatedKey(
        station_from=tariff_key.station_from,
        station_to=tariff_key.station_to,
        tariff_type=tariff_key.tariff_type,
        company=tariff_key.company
    )


def _get_segments_by_undated_keys(segments):
    # type: (List[RThreadSegment]) -> (Dict[TariffKey, List[RThreadSegment]], Dict[TariffUndatedKey, Dict[TariffKey, List[RThreadSegment]]])

    segments_by_keys = defaultdict(list)
    segments_by_undated_keys = defaultdict(lambda: defaultdict(list))

    for segment in segments:
        tariff_key = get_tariff_key(segment)
        if tariff_key:
            undated_key = _get_undated_key(tariff_key)
            segments_by_keys[tariff_key].append(segment)
            segments_by_undated_keys[undated_key][tariff_key].append(segment)

    return segments_by_keys, segments_by_undated_keys


def _get_tariff_types_by_route_numbers(segments):
    # type: (List[RThreadSegment]) -> Dict[str, int]

    numbers_dict = {}
    for segment in segments:
        tariff_type_id = TariffType.DEFAULT_ID
        if segment.thread.tariff_type_id:
            tariff_type_id = segment.thread.tariff_type_id
        elif segment.thread.express_type in [ExpressType.AEROEXPRESS, ExpressType.EXPRESS]:
            tariff_type_id = TariffType.EXPRESS_ID

        if segment.thread.number and segment.thread.company.id in IM_COMPANIES:
            # В общем случае thread.number может содержать список номеров рейсов электричек через слеш
            # Иногда номера могут также включать буквы, которые нам не нужны, оставляем только цифры
            numbers_str = ''.join(char for char in segment.thread.number if char.isdigit() or char == '/')
            numbers = numbers_str.split('/')
            numbers_dict.update({number: tariff_type_id for number in numbers})
    return numbers_dict


@suburban_selling_breaker
@traced_function
def get_suburban_selling_tariffs(
    tariff_keys, selling_flows=None, selling_barcode_presets=None, tariff_type_ids_by_numbers=None
):
    # type: (Iterable[TariffKey], List[str], List[str], Dict[str, int]) -> (List[Dict], List[Dict], List[Dict])
    """
    Получение тарифов из suburban_selling по указанным ключам
    """
    selling_tariffs = []
    result_keys = []
    selling_partners = []
    if not tariff_keys or not selling_flows:
        return selling_tariffs, result_keys, selling_partners

    suburban_tariffs_url = '{}/get_tariffs'.format(settings.SUBURBAN_SELLING_URL)
    data = {
        'keys': [k._asdict() for k in tariff_keys],
        'selling_flows': selling_flows,
        'barcode_presets': selling_barcode_presets,
        'tariff_types': tariff_type_ids_by_numbers
    }
    log.info('Get suburban selling tariffs. Request data: {}'.format(repr(data)))

    try:
        response = requests.post(
            suburban_tariffs_url,
            data=json.dumps(data),
            timeout=settings.SUBURBAN_SELLING_API_TIMEOUT,
        ).json()

        if response['errors']:
            raise Exception(response)

        selling_tariffs = response['result'].get('selling_tariffs')
        result_keys = response['result'].get('keys')
        selling_partners = response['result'].get('selling_partners')

    except Exception:
        log.exception('Get selling tariffs error. Request data: {}'.format(repr(data)))

    return selling_tariffs, result_keys, selling_partners


def _get_tariff_by_ids(selling_tariffs):
    # type: (List[Dict]) -> Dict[int, Dict]

    tariff_by_ids = {}
    for provider in selling_tariffs:
        for tariff in provider['tariffs']:
            tariff['provider'] = provider['provider']
            tariff_by_ids[tariff['id']] = tariff
    return tariff_by_ids


def _add_selling_info_to_segments(segments_by_keys, tariff_by_ids, tariff_keys):
    # type: (Dict[TariffKey, List[RThreadSegment]], Dict[int, Dict], List[Dict]) -> None
    """
    Добавляет selling_info в каждый подходящий сегмент
    Старое поведение при selling=true
    """
    for selling_key in tariff_keys:
        tariff_key = TariffKey(**selling_key['key'])
        for segment in segments_by_keys[tariff_key]:
            segment_provider = 'aeroexpress'

            if selling_key['tariff_ids']:
                segment_tariffs = []
                for tid in selling_key['tariff_ids']:
                    tariff = tariff_by_ids[tid].copy()
                    if tariff.get('provider') == 'movista':
                        segment_provider = 'movista'
                    tariff.pop('id')
                    tariff.pop('provider')
                    segment_tariffs.append(tariff)

                if segment_tariffs:
                    segment.selling_info = {
                        'type': segment_provider,
                        'tariffs': segment_tariffs
                    }


def _add_selling_tariffs_to_search_result(search_result, tariff_by_ids):
    # type: (Dict, Dict[int, Dict]) -> None
    """
    Добавляет selling_tariffs в ответ поиска
    """
    result_tariffs = []
    for tariff in tariff_by_ids.values():
        result_tariffs.append(tariff)
    search_result['selling_tariffs'] = result_tariffs


def _add_selling_tariffs_to_segments(segments_by_undated_keys, tariffs_by_ids, tariff_keys, all_segments):
    # type: (Dict[TariffUndatedKey, Dict[str, List[RThreadSegment]]], Dict[int, Dict], List[Dict], List[RThreadSegment]) -> None
    """
    Добавляет индексы тарифов в каждый подходящий сегмент
    """
    tariff_ids_by_keys = defaultdict(list)
    for selling_key in tariff_keys:
        tariff_key = TariffKey(**selling_key['key'])
        for tariff_id in selling_key['tariff_ids']:
            tariff_ids_by_keys[tariff_key].append(tariff_id)

    for undated_key, segments_by_keys in segments_by_undated_keys.items():
        for _, segments in segments_by_keys.items():
            valid_from_by_tariff_ids = {}
            valid_until_by_tariff_ids = {}
            for tariff_key in segments_by_keys:
                for tariff_id in tariff_ids_by_keys[tariff_key]:
                    tariff = tariffs_by_ids[tariff_id]
                    valid_from_by_tariff_ids[tariff_id] = parser.parse(tariff['valid_from'])
                    valid_until_by_tariff_ids[tariff_id] = parser.parse(tariff['valid_until'])

            for segment in segments:
                segment.selling_tariffs_ids = set()
                for tariff_id in valid_from_by_tariff_ids:
                    if valid_from_by_tariff_ids[tariff_id] <= segment.departure < valid_until_by_tariff_ids[tariff_id]:
                        segment.selling_tariffs_ids.add(tariff_id)

                segment.selling_tariffs_ids = list(segment.selling_tariffs_ids)

    for segment in all_segments:
        if not hasattr(segment, 'selling_tariffs_ids'):
            segment.selling_tariffs_ids = []


def add_suburban_selling_tariffs(
    direct_segments, all_segments, search_result, selling_version=SELLING_V2, selling_flows=None, selling_barcode_presets=None
):
    # type: (List[RThreadSegment], List[RThreadSegment], Dict, str, List[str], List[str]) -> None
    """
    Получение и добавление в ответ информации о тарифах для продажи билетов на электрички
    """
    try:
        segments_by_keys, segments_by_undated_keys = _get_segments_by_undated_keys(direct_segments)

        if selling_version == SELLING_V2:
            selling_flows = [SuburbanSellingFlow.AEROEXPRESS, SuburbanSellingFlow.VALIDATOR]

        tariff_type_ids_by_numbers = (
            _get_tariff_types_by_route_numbers(direct_segments)
            if selling_version == SELLING_V3 else {}
        )

        selling_tariffs, tariff_keys, selling_partners = get_suburban_selling_tariffs(
            segments_by_keys.keys(), selling_flows, selling_barcode_presets, tariff_type_ids_by_numbers
        )

        tariffs_by_ids = _get_tariff_by_ids(selling_tariffs)

        search_result['selling_partners'] = selling_partners

        if selling_version == SELLING_V2:
            _add_selling_info_to_segments(segments_by_keys, tariffs_by_ids, tariff_keys)

        _add_selling_tariffs_to_search_result(search_result, tariffs_by_ids)
        _add_selling_tariffs_to_segments(segments_by_undated_keys, tariffs_by_ids, tariff_keys, all_segments)

    except Exception as ex:
        log.exception('Не смогли получить цены на электрички: {}'.format(repr(ex)))
