# coding: utf8

import logging
import json

from django.conf import settings
from django.utils.encoding import smart_unicode
from rest_framework.decorators import APIView
from rest_framework import status
from rest_framework.decorators import api_view
from rest_framework.response import Response

from common.utils.jsonutils import rich_serialize

from travel.rasp.export.export.v3.core.errors import UserUseError


log = logging.getLogger(__name__)


class BaseView(APIView):
    http_method_names = ['get']

    def http_method_not_allowed(self, request, *args, **kwargs):
        error = UserUseError(u'Использован неподдерживаемый тип запроса - {}. '
                             u'Допустимые типы: {}.'.format(request.method, u''.join(self.http_method_names).upper()),
                             status.HTTP_400_BAD_REQUEST)
        return gen_error_response(request, error)

    def get(self, request, *args, **kwargs):
        return self.make_response(self.handle, request, *args, **kwargs)

    def post(self, request, *args, **kwargs):
        return self.make_response(self.handle, request, *args, **kwargs)

    def handle(self, *args, **kwargs):
        raise NotImplementedError()

    def make_response(self, handle, request, *args, **kwargs):
        http_code = kwargs.get('http_code', status.HTTP_200_OK)
        try:
            result = handle(request, *args, **kwargs)
            if not isinstance(result, dict):
                raise ValueError(u'Неправильный тип ответа: {}\n{}\n'.format(
                    smart_unicode(repr(result)),
                    smart_unicode(repr(request))
                ))
        except UserUseError as ex:
            return gen_error_response(request, ex)
        except Exception as ex:
            if settings.DEBUG:
                raise

            log.error(
                u'Неизвестная ошибка: %s\n%s\n', repr(ex), smart_unicode(repr(request)),
                exc_info=True,
                extra={'request': request}  # for raven
            )
            return gen_error_response(request, UserUseError(u'Неизвестная ошибка: {}.'.format(repr(ex)), 500))
        return json_response(result, http_code)


@api_view(['GET', 'POST'])
def error_view(request, error):
    return gen_error_response(request, error)


def json_response(result, status_code=status.HTTP_200_OK):
    json_str = json.dumps(result, ensure_ascii=False, encoding='utf8', default=rich_serialize)
    result = json.loads(json_str)
    response = Response(result, status=status_code, content_type='application/json; charset=utf-8')
    response['X-Content-Type-Options'] = 'nosniff'
    return response


def gen_error_response(request, error):
    return json_response({'error': {
        'text': error.message,
        'status_code': error.status_code,
        'request': request.build_absolute_uri()}}, error.status_code)
