# coding: utf8
from __future__ import unicode_literals, absolute_import, division, print_function

import json
import pytz
from dateutil.relativedelta import relativedelta
from iso8601 import parse_date, ParseError
from marshmallow import Schema, fields, ValidationError
from marshmallow.validate import Range
from rest_framework import status

from travel.rasp.library.python.common23.date import environment
from common.xgettext.i18n import gettext
from common.serialization.common_schemas import MultiValueDictSchemaMixin

from travel.rasp.export.export.v3.core.errors import UserUseError
from travel.rasp.export.export.v3.core.search import TransfersMode
from travel.rasp.export.export.v3.views.utils import esr_to_id

UP_TO_HOURS_MIN, UP_TO_HOURS_MAX = 0, 12
DAYS_AHEAD_MIN, DAYS_AHEAD_MAX = 0, 100


class DateField(fields.Field):
    SPECIAL_DATES = ['today', 'tomorrow']

    def deserialize(self, value, attr=None, data=None):
        dt_str = value.strip()
        if not dt_str:
            raise ValidationError(gettext('В запросе не указана дата.'), http_code=400)

        if dt_str in DateField.SPECIAL_DATES:
            return dt_str

        try:
            dt = parse_date(dt_str, default_timezone=None)
        except ParseError:
            try:
                space_count = dt_str.count(' ')
                dt = parse_date(dt_str.replace(' ', '+', space_count))
            except Exception:
                raise ValidationError(gettext('Дата должна быть в формате ISO 8601'))

        start_date = (environment.now_aware() - relativedelta(months=11))
        end_date = (environment.now_aware() + relativedelta(months=11))

        if ((dt.tzinfo and not (start_date.astimezone(dt.tzinfo) <= dt <= end_date.astimezone(dt.tzinfo))) or
                (not start_date.date() <= dt.date() <= end_date.date())):
            raise ValidationError(gettext('Указана недопустимая дата - {}. Доступен выбор даты на 30 дней назад и '
                                          '11 месяцев вперед от текущей даты'.format(dt.date())), http_code=400)
        return dt


class TimezoneField(fields.Field):
    def deserialize(self, value, attr=None, data=None):
        timezone_str = value.strip()
        if not timezone_str:
            return

        try:
            timezone = pytz.timezone(timezone_str)
        except pytz.exceptions.UnknownTimeZoneError:
            raise ValidationError(gettext('Указана неподдерживаемая таймзона.'))
        return timezone


class BaseRequestSchema(Schema):
    def handle_error(self, exceptions, data):
        return gen_validate_error(exceptions.messages)

    def get_str_value(self, value):
        return value.strip()

    def get_bool_param(self, value):
        if isinstance(value, basestring):
            return value.lower() == 'true'
        return value

    def get_station_id(self, value):
        value = self.get_str_value(value.strip())
        return esr_to_id(value, hidden=True)


class BaseSearchRequestSchema(BaseRequestSchema):
    timezone = TimezoneField(missing='')
    city_from_id = fields.Method(deserialize='get_str_value', missing=None, load_from='city_from')
    city_to_id = fields.Method(deserialize='get_str_value', missing=None, load_from='city_to')
    station_from_id = fields.Method(deserialize='get_station_id', missing=None, load_from='station_from')
    station_to_id = fields.Method(deserialize='get_station_id', missing=None, load_from='station_to')


class SearchOnDateRequestSchema(BaseSearchRequestSchema, MultiValueDictSchemaMixin):
    tomorrow_upto = fields.Integer(
        missing=0,
        validate=Range(UP_TO_HOURS_MIN, UP_TO_HOURS_MAX,
                       gettext('Параметр up_to должен быть положительным числом в интервале {} {},'
                               ' по умолчанию 0').format(UP_TO_HOURS_MIN, UP_TO_HOURS_MAX)))
    days_ahead = fields.Integer(
        missing=1,
        validate=Range(DAYS_AHEAD_MIN, DAYS_AHEAD_MAX,
                       gettext('Параметр days_ahead должен быть положительным числом в интервале {} {},'
                               ' по умолчанию 1').format(DAYS_AHEAD_MIN, DAYS_AHEAD_MAX)))

    dt = DateField(missing='', load_from='date')
    transfers = fields.String(missing=TransfersMode.NONE)
    selling = fields.Method(deserialize='get_bool_param', missing=False)
    selling_version = fields.String(missing=None)
    selling_flows = fields.List(fields.String(), missing=None)
    selling_barcode_presets = fields.List(fields.String(), missing=None)
    disable_cancels = fields.Method(deserialize='get_bool_param', missing=True)


class ThreadRequestSchema(BaseRequestSchema):
    station_from_id = fields.Method(deserialize='get_station_id', missing=None, load_from='station_from')
    station_to_id = fields.Method(deserialize='get_station_id', missing=None, load_from='station_to')


class ThreadOnDateRequestSchema(ThreadRequestSchema):
    dt = DateField(missing='', load_from='date')
    disable_cancels = fields.Method(deserialize='get_bool_param', missing=True)


def gen_validate_error(errors):
    raise UserUseError(json.dumps(errors, ensure_ascii=False), status.HTTP_400_BAD_REQUEST)
