# coding: utf8
from __future__ import unicode_literals, absolute_import, division, print_function

import logging
import pytz

from travel.rasp.export.export.v3.selling.suburban import SELLING_V1
from travel.rasp.export.export.v3.core.search import search_on_all_days, search_on_date
from travel.rasp.export.export.v3.views.base_view import BaseView
from travel.rasp.export.export.v3.views.request_schemas import BaseSearchRequestSchema, SearchOnDateRequestSchema
from travel.rasp.export.export.v3.views.utils import get_points, get_today_tomorrow_date, dt_to_tz_date


log = logging.getLogger(__name__)


class SearchView(BaseView):
    def handle(self, request, *args, **kwargs):
        query, errors = BaseSearchRequestSchema(context={'request': request}).load(request.GET)
        point_from, point_to, point_from_reduced, point_to_reduced = get_points(
            query['station_from_id'], query['station_to_id'], query['city_from_id'], query['city_to_id']
        )
        national_version, lang = request.national_version, request.language_code
        search_result = search_on_all_days(
            request, point_from, point_to, point_from_reduced, point_to_reduced,
            query['timezone'], national_version=national_version, lang=lang
        )
        return search_result


class SearchOnDateView(SearchView):
    def handle(self, request, *args, **kwargs):
        query, errors = SearchOnDateRequestSchema(context={'request': request}).load(request.GET)
        point_from, point_to, point_from_reduced, point_to_reduced = get_points(
            query['station_from_id'], query['station_to_id'], query['city_from_id'], query['city_to_id']
        )
        dt = query['dt']
        from_tz = pytz.timezone(point_from.time_zone)
        date_ = get_today_tomorrow_date(dt, from_tz)
        if not date_:
            date_ = dt_to_tz_date(dt, point_from.pytz)

        national_version, lang = request.national_version, request.language_code
        selling_version = query['selling_version'] or (SELLING_V1 if query['selling'] else None)

        search_result = search_on_date(
            request, point_from, point_to, point_from_reduced, point_to_reduced,
            query['timezone'], date_, query['days_ahead'], query['tomorrow_upto'],
            selling_version=selling_version,
            selling_flows=query['selling_flows'], selling_barcode_presets=query['selling_barcode_presets'],
            national_version=national_version, lang=lang, transfers_mode=query['transfers'],
            disable_cancels=query['disable_cancels']
        )
        return search_result
