# coding: utf-8

import inspect
import hashlib

from django.utils.translation import get_language
from rest_framework.views import APIView

from common.utils.httpcaching import modified_by_timestamp_cached

from travel.rasp.export.export.v3.core.station import get_station_info, get_station_types
from travel.rasp.export.export.v3.views.base_view import BaseView
from travel.rasp.export.export.v3.views.request_schemas import BaseRequestSchema
from travel.rasp.export.export.v3.views.utils import get_station_by_esr_or_404


STATION_TYPES_CACHE_TIMEOUT = 2 * 7 * 24 * 3600  # 2 weeks


class StationRequestSchema(BaseRequestSchema):
    pass


class StationView(BaseView):
    def handle(self, request, esr_code, *args, **kwargs):
        query, errors = StationRequestSchema(context={'request': request}).load(request.GET)
        station = get_station_by_esr_or_404(esr_code, hidden=True)
        return get_station_info(station)


class StationTypesView(BaseView):
    def handle(self, request, *args, **kwargs):
        return {'station_types' : get_station_types()}


def station_types_hash(func, request):
    key = '{}:{}:{}:{}'.format(
        inspect.getfile(func),
        func.__name__,
        request.path,
        get_language() + request.national_version)

    return hashlib.md5(key).hexdigest()


class StationTypesProxyView(APIView):
    def get(self, request, *args, **kwargs):
        @modified_by_timestamp_cached('prepare_all', timeout=STATION_TYPES_CACHE_TIMEOUT, hash_func=station_types_hash)
        def dec_func(req):
            view = StationTypesView.as_view()
            return view(req._request, *args, **kwargs).render()
        return dec_func(request)
