# coding: utf8

import pytz

from travel.rasp.export.export.v3.core.station_schedule import get_station_schedule_on_date, get_station_schedule_on_all_days
from travel.rasp.export.export.v3.views.base_view import BaseView
from travel.rasp.export.export.v3.views.request_schemas import BaseRequestSchema, TimezoneField, DateField
from travel.rasp.export.export.v3.views.utils import get_station_by_esr_or_404, get_today_tomorrow_date, dt_to_tz_date


class StationScheduleRequestSchema(BaseRequestSchema):
    timezone = TimezoneField(missing=u'')


class StationScheduleOnDateRequestSchema(StationScheduleRequestSchema):
    dt = DateField(missing=u'', load_from='date')


class StationScheduleView(BaseView):
    def handle(self, request, esr_code, *args, **kwargs):
        query, errors = StationScheduleRequestSchema(context={'request': request}).load(request.GET)
        station = get_station_by_esr_or_404(esr_code, hidden=True)
        national_version, lang = request.national_version, request.language_code
        return get_station_schedule_on_all_days(station, esr_code, query['timezone'],
                                                national_version=national_version, lang=lang)


class StationScheduleOnDateView(BaseView):
    def handle(self, request, esr_code, *args, **kwargs):
        query, errors = StationScheduleOnDateRequestSchema(context={'request': request}).load(request.GET)
        station = get_station_by_esr_or_404(esr_code, hidden=True)

        dt = query['dt']
        from_tz = pytz.timezone(station.time_zone)
        request_date = get_today_tomorrow_date(dt, from_tz)
        if not request_date:
            request_date = dt_to_tz_date(dt, from_tz)

        national_version, lang = request.national_version, request.language_code
        return get_station_schedule_on_date(station, esr_code, query['timezone'], request_date,
                                            national_version=national_version, lang=lang)
