# coding: utf8
from __future__ import unicode_literals, absolute_import, division, print_function

from datetime import datetime, time, timedelta

from django.http import Http404
from rest_framework import status

import pytz

from common.models.geo import Station, Settlement
from common.models.schedule import RThread
from travel.rasp.library.python.common23.date.environment import now_aware
from common.xgettext.i18n import gettext as _

from geosearch.views.pointlist import PointList
from geosearch.views.pointtopoint import process_points_lists, SamePointError

from travel.rasp.export.export.v3.core.errors import UserUseError


def append_station_title_prefix(station, title):
    if station.t_type_id == 2:
        return _(u'аэропорт %s') % title
    else:
        return title


def get_server_time():
    return now_aware().astimezone(pytz.utc).isoformat()


def get_day_start_end_utc_in_iso(day, tz):
    start_day = tz.localize(datetime.combine(day, time())).astimezone(pytz.utc).isoformat()
    end_day = tz.localize(datetime.combine(day, time(23, 59, 59))).astimezone(pytz.utc).isoformat()
    return start_day, end_day


def get_today_tomorrow_date(date_str, from_tz):
    if date_str in ['today', 'tomorrow']:
        msk_dt = now_aware()
        if date_str == 'tomorrow':
            msk_dt += timedelta(1)
        return msk_dt.astimezone(from_tz).date()


def esr_to_id(esr_code, hidden=False):
    try:
        return get_station_by_esr_or_404(esr_code, hidden).id

    except Http404:
        return None


def get_point(station_id, city_id):
    if station_id:
        try:
            return Station.objects.get(id=station_id)
        except Station.DoesNotExist:
            pass

    if city_id:
        try:
            return Settlement.objects.get(id=city_id)
        except Settlement.DoesNotExist:
            pass

    return None


def get_points(station_from_id, station_to_id, city_from_id, city_to_id):
    point_from = get_point(station_from_id, city_from_id)
    point_to = get_point(station_to_id, city_to_id)

    if point_from is None or point_to is None:
        raise UserUseError(u'Указана неизвестная точка отправления или прибытия.')

    point_list_from = PointList(point_from, [point_from], None, exact_variant=True)
    point_list_to = PointList(point_to, [point_to], None, exact_variant=True)

    point_from_reduced, point_to_reduced = isinstance(point_from, Settlement), isinstance(point_to, Settlement)

    try:
        point_from, point_to = process_points_lists(point_list_from, point_list_to, suburban=True)
    except SamePointError:
        raise UserUseError(u'Точка прибытия совпадает с точкой отправления.')

    point_from_reduced &= isinstance(point_from.point, Station)
    point_to_reduced &= isinstance(point_to.point, Station)

    return point_from.point, point_to.point, point_from_reduced, point_to_reduced


def get_station_by_esr_or_404(esr_code, hidden=False):
    try:
        if hidden:
            return Station.hidden_manager.get(code_set__system__code__iexact='esr',
                                              code_set__code__iexact=esr_code)
        else:
            return Station.get_by_code('esr', esr_code)

    except Station.DoesNotExist:
        raise UserUseError(u'Станции с кодом esr={} нет в базе'.format(esr_code), status.HTTP_404_NOT_FOUND)


def dt_to_tz(dt, tz):
    if not dt:
        return
    if dt.tzinfo:
        return dt.astimezone(tz)
    return tz.localize(dt)


def dt_to_tz_date(dt, tz):
    dt = dt_to_tz(dt, tz)
    if isinstance(dt, datetime):
        return dt.date()


def try_get_thread(uid):
    try:
        thread = RThread.objects.get(uid=uid)
    except RThread.DoesNotExist:
        raise UserUseError(u'Рейса с uid={} нет в базе'.format(uid), status.HTTP_404_NOT_FOUND)
    return thread
