# coding: utf8

import hashlib
import inspect
import json
import logging
from datetime import datetime

from django.conf import settings
from django.http import Http404
from django.shortcuts import get_object_or_404
from django.utils.translation import get_language

from rest_framework import status
from rest_framework.views import APIView
from marshmallow import fields

from common.models.geo import SuburbanZone
from travel.rasp.library.python.common23.date import environment
from common.utils.caching import get_package_cache_root
from common.utils.httpcaching import modified_by_timestamp_cached, call_hash

from travel.rasp.export.export.v3.core.errors import UserUseError
from travel.rasp.export.export.v3.core.zones import get_all_zones, get_zone
from travel.rasp.export.export.v3.views.base_view import BaseView
from travel.rasp.export.export.v3.views.request_schemas import BaseRequestSchema


log = logging.getLogger(__name__)


CACHE_TIMEOUT = 2 * 7 * 24 * 3600  # 2 weeks


class ZonesRequestSchema(BaseRequestSchema):
    use_directions = fields.Method(deserialize='get_bool_param', missing=False)


class ZoneRequestSchema(BaseRequestSchema):
    pass


def log_response_time(func):
    def wrapper(request):
        start_time = datetime.now()
        result = func(request)
        diff = (datetime.now() - start_time).total_seconds()
        if diff > 3:
            log.info(
                u'Ручка zone/zones отработала дольше чем за 1 секунду\n'
                u'path: {}\nfull_url: {}\nuser_agent: {}\ndc: {}\nkey: {}\ndt: {}\ntime_diff: {}'.format(
                    request.path,
                    request.build_absolute_uri(),
                    request.META.get('HTTP_USER_AGENT'),
                    json.dumps(settings.CACHES['default']),
                    '{}httpcaching.py/{}'.format(get_package_cache_root(), call_hash(func, request)),
                    environment.now(),
                    diff)
            )
        return result
    return wrapper


def zones_hash(func, request):
    params = {
        'lang': get_language(),
        'national_version': request.national_version,
        'use_directions': request.GET.get('use_directions') == 'true'
    }
    s = '{}:{}:{}:{}'.format(inspect.getfile(func), func.__name__, request.path, str(sorted(params.items())))

    return hashlib.md5(s).hexdigest()


class ProxyView(APIView):
    """
    Для использования декоратора modified_by_timestamp_cached
    нужно отрендерить ответ. По умолчанию метод render вызывается после того как вью вернет ответ.
    Поэтому в классе происходит вызов нужного вью и рендер ответа.
    """
    view_class = None

    def get(self, request, *args, **kwargs):
        @log_response_time
        @modified_by_timestamp_cached('prepare_all', timeout=CACHE_TIMEOUT, hash_func=zones_hash)
        def dec_func(req):
            view = self.view_class.as_view()
            return view(req._request, *args, **kwargs).render()
        return dec_func(request)


class ZoneView(BaseView):
    def handle(self, request, zone_id, *args, **kwargs):
        query, errors = ZoneRequestSchema(context={'request': request}).load(request.GET)
        try:
            zone = get_object_or_404(SuburbanZone, id=zone_id)
        except Http404 as ex:
            raise UserUseError(ex.message, status.HTTP_404_NOT_FOUND)
        return get_zone(zone)


class ZonesView(ZoneView):
    def handle(self, request, *args, **kwargs):
        query, errors = ZonesRequestSchema(context={'request': request}).load(request.GET)
        return get_all_zones(use_directions=query['use_directions'])


class ZoneProxyView(ProxyView):
    view_class = ZoneView


class ZonesProxyView(ProxyView):
    view_class = ZonesView
