# -*- coding: utf-8 -*-
from __future__ import absolute_import

import json
from collections import defaultdict

from django.http import HttpResponse

from common.models.geo import Station, StationCode, CodeSystem
from common.models.transport import TransportType
from common.models_utils import fetch_related


def get_airports():
    plane_t_type = TransportType.objects.get(code='plane')
    stations = list(
        Station.objects.filter(hidden=False, t_type=plane_t_type)
        .order_by('title')
    )
    fetch_related(stations, 'settlement', 'country', model=Station)
    stations_by_id = {s.id: s for s in stations}

    # префетчим нужные CodeSystem
    code_systems_names = ['iata', 'icao']
    code_systems_by_id = {
        cs.id: cs
        for cs in CodeSystem.objects.filter(code__in=code_systems_names)
    }

    # префетчим нужные StationCode
    station_codes = StationCode.objects.filter(
        station_id__in=stations_by_id.keys(),
        system_id__in=code_systems_by_id.keys()
    )

    station_codes_by_station = defaultdict(dict)
    for code in station_codes:
        station = stations_by_id[code.station_id]
        code_system = code_systems_by_id[code.system_id]
        station_codes_by_station[station][code_system.code] = code.code

    airports = []
    for station in stations:
        airport_data = {
            'title': station.title,
            'title_ru': station.title_ru,
            'title_en': station.title_en,
            'time_zone': station.time_zone
        }

        for code_system in code_systems_names:
            station_code = station_codes_by_station[station].get(code_system)

            if station_code:
                airport_data[code_system] = station_code

        for field, point in (('city', station.settlement), ('country', station.country)):
            if not point:
                continue

            airport_data[field] = {
                'title': point.title,
                'title_ru': point.title_ru,
                'title_en': point.title_en,
                'geo_id': point._geo_id
            }

        airports.append(airport_data)

    return airports


def airports(request):
    airports_data = get_airports()

    return HttpResponse(
        json.dumps(
            {"airports": airports_data},
            ensure_ascii=False,
            separators=(',', ':')
        ),
        content_type="application/json; charset=utf-8"
    )
