# -*- coding: utf-8 -*-

import os

from xml.etree import ElementTree

from django.conf import settings
from django.http import HttpResponse
from django.shortcuts import get_object_or_404

from common.models.geo import Station, Settlement, Region, StationCode
from common.utils.httpcaching import modified_by_prepare_all
from common.utils.httpresponses import xmlresponse

from travel.rasp.export.export.views.utils import append_station_title_prefix


@modified_by_prepare_all
@xmlresponse
def city_directions(request, city_id):
    city = get_object_or_404(Settlement, id=city_id)

    tree = ElementTree.Element('citydirections')

    if not city.suburban_zone:
        return tree

    for d in city.suburban_zone.externaldirection_set.all():
        direction = ElementTree.Element(
            'direction',
            code=d.code,
            title=d.title,
        )

        tree.append(direction)

        stations = list(Station.objects.filter(externaldirectionmarker__external_direction=d))

        id_to_code = StationCode.id_to_code_dict_for('esr', stations)

        for s in stations:
            try:
                esr_code = id_to_code[s.id]
            except KeyError:
                pass

            station = ElementTree.Element(
                'station',
                esr=esr_code,
                title=s.title,
                importance=str(s.majority_id)
            )

            direction.append(station)

    return tree


@modified_by_prepare_all
@xmlresponse
def city_stations(request, city_id):
    u"""RASP-4422"""
    city = get_object_or_404(Settlement, id=city_id)

    if city.suburban_zone:
        return export_suburban_stations(city.suburban_zone)

    else:
        tree = ElementTree.Element('citystations')
        stations = list(city.station_set.filter(hidden=False))

        export_stations_list(tree, stations, city=city)

        return tree


def export_suburban_stations(suburban_zone):
    filename = os.path.join(settings.EXPORT_SUBURBAN_PATH, 'suburban_zone_%d.xml' % suburban_zone.id)

    if os.path.exists(filename):  # Есть уже посчитанный
        try:
            return HttpResponse(open(filename).read(), content_type='application/xml')

        except IOError:
            pass

    return HttpResponse(get_suburban_zone_xml(suburban_zone), content_type='application/xml')


def get_suburban_zone_xml(suburban_zone):
    tree = ElementTree.Element('citystations')
    markers = suburban_zone.markers()
    stations = Station.objects.in_bulk(marker.station_id for marker in markers).values()
    export_stations_list(tree, stations, markers)
    xml = ElementTree.tostring(tree, 'UTF-8')

    return xml


def export_stations_list(tree, stations, markers=None, city=None, add_station_codes=None, stations_ids_by_title=None):
    id_to_code = StationCode.id_to_code_dict_for('esr', stations)
    visited_stations = {}

    regions = Region.objects.in_bulk(station.region_id for station in stations)
    settlements = Settlement.objects.in_bulk(station.settlement_id for station in stations)

    for station in stations:
        try:
            esr_code = id_to_code[station.id]
        except KeyError:
            continue

        if esr_code in visited_stations:
            continue

        if not station.has_suburban_traffic():
            continue

        title = station.L_title()
        popular_title = station.L_popular_title(fallback=False)

        title = append_station_title_prefix(station, title)

        direction = get_station_direction(station, markers, city)

        branch = ElementTree.Element(
            'station',
            esr=esr_code,
            title=title or u'',
            popular_title=popular_title or u'',
            importance=str(station.majority_id),
            region=regions[station.region_id].L_title() if station.region_id in regions else "",
            direction=direction.L_title() if direction else ""

        )

        if station.settlement_id in settlements:
            branch.set('city', settlements[station.settlement_id].L_title())

        if station.latitude:
            branch.set('lat', '%.6f' % station.latitude)
            branch.set('lon', '%.6f' % station.longitude)

        if station.L_short_title():
            branch.set('short_title', station.L_short_title())

        if add_station_codes:
            branch.set('yandex_rasp_code', str(station.id))

        if stations_ids_by_title:
            branch.set(
                'need_direction', 'true'
                if (len(stations_ids_by_title.get(station.region, {}).get(station.title, [])) > 1)
                else 'false'
            )

        tree.append(branch)
        visited_stations[esr_code] = True


def get_station_direction(station, markers, city):
    if markers:
        for marker in markers:
            if marker.station_id == station.pk:
                return marker.external_direction

    else:
        try:
            return station.externaldirectionmarker_set.filter(
                external_direction__suburban_zone__settlement=city
            ).select_related('external_direction')[0].external_direction

        except IndexError:
            pass

    return None
