# -*- coding: utf-8 -*-

from collections import defaultdict
from xml.etree import ElementTree

from django.db.models import Q

from common.models.geo import Station, Region, StationCode, ExternalDirection, ExternalDirectionMarker
from common.utils.httpcaching import modified_by_prepare_all
from common.utils.httpresponses import xmlresponse
from common.xgettext.i18n import gettext as _

# Для некоторых пар национальнaя версия и страна нужно добавлять регионы
ADDITIONAL_REGION_IDS_BY_NATIONAL_VERSION_AND_COUNTRY_ID = defaultdict(list, {
    ('ua', 187): (977, )  # ('ua', Украина) - Крым
})


@modified_by_prepare_all
@xmlresponse
def country_directions(request, country_id):
    country_id = int(country_id)

    directions = get_country_directions(request.national_version, country_id)

    edms = list(ExternalDirectionMarker.objects.filter(external_direction__in=directions)
                .values('external_direction_id', 'station_id', 'order').order_by())

    regions = Region.objects.in_bulk({_d.base_station.region_id for _d in directions})

    region_directions = defaultdict(list)

    for d in directions:
        region_directions[d.base_station.region_id].append(d)

    station_by_direction = defaultdict(list)

    for edm in edms:
        station_by_direction[edm['external_direction_id']].append(edm)

    station_ids = {_edm['station_id'] for _edm in edms}
    station_ids.update({_d.base_station_id for _d in directions})

    station_by_id = Station.objects.order_by().in_bulk(station_ids)
    station_by_id[None] = None

    station_id2esr = dict(
        StationCode.objects.filter(system__code='esr', station__in=station_ids)
                           .values_list('station_id', 'code')
    )

    tree = ElementTree.Element('countrydirections')
    tree.attrib['country_id'] = str(country_id)

    for region_id in regions.keys():
        region = regions[region_id]
        region_el = ElementTree.Element(
            'region',
            id=str(region_id),
            title=region.L_title() or ''
        )

        for direction in region_directions[region_id]:
            direction_el = ElementTree.Element(
                'direction',
                code=str(direction.code),
                title=direction.L_title() or ''
            )

            markers = station_by_direction.get(direction.id, [])
            markers.sort(key=lambda _m: _m['order'])

            for m in markers:
                st = station_by_id[m['station_id']]
                if st.t_type_id != 1:
                    continue

                title = st.L_title() or st.title or ''
                full_title = _(u'Платформа %s') % title if st.station_type_id == 2 else title
                station_el = ElementTree.Element(
                    'station',
                    esr=station_id2esr.get(st.id, u''),
                    title=title,
                    full_title=full_title,
                    longitude=str(st.longitude),
                    latitude=str(st.latitude)
                )
                direction_el.append(station_el)

            region_el.append(direction_el)

        tree.append(region_el)

    return tree


def get_country_directions(national_version, country_id):
    directions = ExternalDirection.objects.filter(
        build_external_directions_query(national_version, country_id)
    ).select_related('base_station', 'base_station__region')

    def is_same_country(station):
        real_country = station.translocal_country(national_version=national_version)

        if real_country is not None and real_country.id != country_id:
            return False

        return True

    return [direction for direction in directions if is_same_country(direction.base_station)]


def build_external_directions_query(national_version, country_id):
    external_directions_query = Q(base_station__country_id=country_id)

    additional_region_ids = \
        ADDITIONAL_REGION_IDS_BY_NATIONAL_VERSION_AND_COUNTRY_ID[(national_version, country_id)]

    if additional_region_ids:
        external_directions_query |= Q(base_station__region_id__in=additional_region_ids)

    return external_directions_query
