# coding: utf8
from __future__ import unicode_literals, absolute_import, division, print_function

from collections import defaultdict
from datetime import datetime
from itertools import chain

from bson import SON
from django.conf import settings
from django.db.models import Q
from mongoengine.context_managers import switch_db

from common.apps.suburban_events import dynamic_params
from common.apps.suburban_events.models import ThreadStationState, EventStateType, SuburbanKey
from common.models.geo import Station
from common.models.schedule import RTStation, TrainTurnover
from common.utils.date import RunMask, MSK_TZ
from travel.rasp.library.python.common23.date.environment import now
from travel.rasp.export.export.v3.views.base_view import BaseView


class CurrentEventsView(BaseView):
    """ Отдает текущие опоздания электричек. """

    def handle(self, request, *args, **kwargs):
        last_query_time = dynamic_params.get_param('last_successful_query_time')

        with switch_db(ThreadStationState, settings.SUBURBAN_EVENTS_DATABASE_NAME + '_no_timeout'):
            docs = ThreadStationState.objects.aggregate(
                {'$match': {
                    '$or': [
                        {'arrival_state.type': EventStateType.POSSIBLE_DELAY},
                        {'departure_state.type': EventStateType.POSSIBLE_DELAY},
                    ],
                    'key.thread_start_date': {
                        '$gte': now().replace(hour=0, minute=0, second=0, microsecond=0)
                    },
                    'outdated': {'$ne': True},
                }},
                {'$sort': SON([('_id', 1)])},
                {'$group': {
                    '_id': {'thread_key': '$key.thread_key', 'start_date': '$key.thread_start_date'},
                    'station_id': {'$first': '$key.station_key'},
                    'departure_state': {'$first': '$departure_state.type'},
                    'arrival_trigger': {'$first': '$arrival_state.trigger'},
                    'departure_trigger': {'$first': '$departure_state.trigger'},
                    'arrival_sub': {'$first': '$arrival_state.sub_type'},
                    'departure_sub': {'$first': '$departure_state.sub_type'},

                }},
            )

            if settings.YANDEX_ENVIRONMENT_TYPE == 'testing':
                admin_url = 'work.admin-test.rasp.yandex-team.ru'
                export_url = 'export.tst.rasp.yandex.net'
            else:
                admin_url = 'work.admin.rasp.yandex-team.ru'
                export_url = 'export.rasp.yandex.net'

            thread_date = defaultdict(list)
            for doc in docs:
                date = doc['_id']['start_date'].date()
                for key in SuburbanKey.objects.filter(key=doc['_id']['thread_key']):
                    if date in RunMask(key.thread.year_days):
                        thread_date[date].append((doc, key.thread))

        rtstations = list(RTStation.objects.filter(
            Q(tz_arrival__isnull=True) | Q(tz_departure__isnull=True),
            thread_id__in=[thread for (doc, thread) in chain(*thread_date.values())],
        ).select_related(
            'thread', 'station'
        ).order_by('id'))

        thread_paths = defaultdict(list)
        for rts in rtstations:
            thread_paths[rts.thread].append(rts)

        threads_by_type = {'no_data': [], 'no_next_data': [], 'by_fact': [], 'train_turnover': []}

        for start_date, threads in thread_date.items():
            for (doc, thread) in threads:
                station = Station.objects.get(id=int(doc['station_id']))
                start_dt = datetime.combine(start_date, thread.tz_start_time)

                if doc.get('departure_sub') == 'no_data_first_station':
                    delay_type = 'no_data'
                    trigger = doc.get('departure_trigger')
                elif doc.get('arrival_sub') == 'no_next_data' or doc.get('departure_sub') == 'no_next_data':
                    delay_type = 'no_next_data'
                    trigger = doc.get('arrival_trigger') or doc.get('departure_trigger')
                elif doc.get('departure_sub') == 'train_turnover':
                    delay_type = 'train_turnover'
                    trigger = doc.get('departure_trigger')
                else:
                    delay_type = 'by_fact'
                    trigger = doc.get('arrival_trigger') or doc.get('departure_trigger')

                data = {
                    'desc': '{} опаздывает нитка {} {} (начиная со станции {})'.format(
                        date,
                        thread.number,
                        thread.title,
                        station.title
                    ),
                    'thread': {
                        'id': thread.id,
                        'title': thread.title,
                        'number': thread.number,
                        'admin_url': 'https://{}/admin/www/rthread/{}'.format(admin_url, thread.id),
                        'export_url': 'https://{}/v3/suburban/thread_on_date/{}/?date={}'.format(
                            export_url,
                            thread.uid,
                            date
                        ),
                        'departure_dt': thread_paths[thread][0].get_event_dt(
                            'departure',
                            start_dt,
                            MSK_TZ
                        ).replace(tzinfo=None).isoformat(),
                        'arrival_dt': thread_paths[thread][1].get_event_dt(
                            'arrival',
                            start_dt,
                            MSK_TZ
                        ).replace(tzinfo=None).isoformat(),
                    },
                    'station': {
                        'id': station.id,
                        'esr': station.get_code('esr'),
                        'title': station.title,
                        'admin_url': 'https://{}/admin/www/station/{}'.format(admin_url, station.id),
                    },
                    'search_example': 'https://{}/v3/suburban/search_on_date/?'
                                      'date={}&station_from={}&station_to={}'.format(
                        export_url, date, station.get_code('esr'), list(thread.path)[-1].station.get_code('esr'))
                }

                if trigger:
                    if delay_type == 'train_turnover':
                        thread_key = trigger
                        for key in SuburbanKey.objects.filter(key=thread_key):
                            if date in RunMask(key.thread.year_days):
                                trigger_thread = key.thread
                                break

                        turnovers = TrainTurnover.objects.filter(
                            number_before=trigger_thread.number,
                            number_after=thread.number,
                            station=thread_paths[thread][0].station,
                        )
                        for turnover in turnovers:
                            if date in RunMask(turnover.year_days):
                                trigger_turnover = turnover
                                break
                        data['trigger'] = {
                            'thread_before_admin_url': 'https://{}/admin/www/rthread/{}'.format(
                                admin_url, trigger_thread.id),
                            'turnover_admin_url': 'https://{}/admin/www/trainturnover/{}'.format(
                                admin_url, trigger_turnover.id),
                        }
                    else:
                        st = Station.objects.get(id=int(trigger))
                        data['trigger_station'] = st.title + ' ' + trigger
                threads_by_type[delay_type].append(data)
        threads_by_type['last_rzd_event_time'] = str(last_query_time)
        return threads_by_type
