# coding: utf8
import logging
import json
import re

from datetime import datetime, timedelta, time
from functools import partial
from itertools import groupby

import pytz
from django.conf import settings
from django.http import HttpResponse, Http404
from django.shortcuts import get_object_or_404
from django.utils.translation import get_language
from xml.etree import ElementTree

from common.models.geo import Station, Settlement
from common.models_utils.i18n import RouteLTitle
from common.models.schedule import RThread, TrainSchedulePlan, StationSchedule, RTStation
from common.models.transport import TransportType, TransportSubtype
from common.models_utils import fetch_related
from common.utils import marketstat
from travel.rasp.library.python.common23.date import environment
from common.utils.date import parse_date, MSK_TZ, RunMask
from common.utils.httpresponses import xmlresponse
from common.xgettext.i18n import gettext as _

from geosearch.views.pointlist import PointList
from geosearch.views.pointtopoint import process_points_lists, SamePointError
from route_search.facilities import fill_suburban_facilities as fill_segments_suburban_facilities
from route_search.shortcuts import search_routes, find
from route_search.tablo import add_z_tablos_to_segments
from stationschedule import SUBURBAN, get_schedule_class
from stationschedule.models import ZTablo2
from stationschedule.facilities import fill_suburban_facilities as fill_schedule_suburban_facilities

from travel.rasp.export.export.views.tariffs import add_suburban_tariffs, build_tariffs_by_stations, set_segment_tariff
from travel.rasp.export.export.views.utils import (
    append_station_title_prefix, add_server_time, get_day_start_end_utc_in_iso, get_today_tomorrow_date,
    get_sub_version
)
from travel.rasp.export.export.teasers import TeaserSetExport, teasers_to_xml

log = logging.getLogger(__name__)
users_search_log = marketstat.DsvLog(settings.USERS_SEARCH_LOG) if settings.USERS_SEARCH_LOG else None


def add_departure_delays_data(xml_el, ztablo, debug=False, thread_uid=None):
    if debug:  # TODO: remove debug mode when real delays data will be in db
        if hash(thread_uid) % 2:
            delay, comment, canceled = '0', '', True
        else:
            delay, comment, canceled = '42', 'Chaos and anarchy!', False
    else:
        if ztablo:
            if ztablo.real_departure and ztablo.departure:
                delay_seconds = int((ztablo.real_departure - ztablo.departure).total_seconds())
                delay = str(delay_seconds / 60)
            else:
                delay = ''
            comment = ztablo.L_comment()
            canceled = ztablo.departure_cancelled
        else:
            delay, comment, canceled = '', '', False

    xml_el.attrib['delay_minutes'] = delay
    xml_el.attrib['delay_comment'] = comment
    xml_el.attrib['canceled'] = 'true' if canceled else 'false'


def add_facilities_data(xml_el, facilities):
    if not facilities:
        return

    facilities_el = ElementTree.Element('facilities')
    for facility in (facilities or []):
        facility_el = ElementTree.Element(
            'facility',
            attrib={
                'code': facility.code,
                'title': facility.L_title(),
                'icon': facility.icon.name and facility.icon.url
            })
        facilities_el.append(facility_el)

    xml_el.append(facilities_el)


def get_default_stops(export_version):
    return _(u'остановки по списку') if export_version == '1' else u''


def station_schedule_base(request, esr_code, **kwargs):
    now_aware = environment.now_aware()

    add_delays = kwargs.pop('add_delays', False)
    export_version = kwargs.pop('version', '1')
    export_sub_version = kwargs.get('sub_version', 1)

    station = get_station_by_esr_or_404(esr_code, hidden=True)
    station_el = ElementTree.Element('station')
    if esr_code:
        station_el.attrib['esr'] = esr_code

    # Расписание на заданный день
    date_str = request.GET.get('date')
    from_tz = pytz.timezone(station.time_zone)
    date_ = get_today_tomorrow_date(date_str, from_tz)

    if not date_:
        try:
            date_ = parse_date(date_str)
        except (ValueError, TypeError):
            date_ = None

    # Формируем расписание
    schedule_cls = get_schedule_class(station, schedule_type=SUBURBAN, t_type_code=SUBURBAN)
    schedule = schedule_cls(station, requested_direction='all', schedule_date=date_)

    msk_today = now_aware.astimezone(MSK_TZ).date()
    current_plan, next_plan = schedule.current_next_plans(msk_today)

    schedule.build(schedule_date=date_)

    # Пропускаем нитки отмены
    schedule_routes = [schedule_route for schedule_route in schedule.schedule
                       if not schedule_route.cancel]

    # заполняем направления
    schedule.fill_directions(schedule_routes)

    direction_title_by_code = schedule.direction_title_by_code

    RouteLTitle.fetch([schedule_route.thread.L_title for schedule_route in schedule_routes])

    if add_delays:
        schedule.fetch_z_tablos(schedule_routes)

    if date_ and export_version != '1':
        fill_schedule_suburban_facilities(schedule_routes)
        if export_sub_version >= 2.1:
            day_start_utc_iso, day_end_utc_iso = get_day_start_end_utc_in_iso(date_, station.pytz)
            station_el.set('day_start_utc', day_start_utc_iso)
            station_el.set('day_end_utc', day_end_utc_iso)
            station_el.set('date', date_.strftime("%Y-%m-%d"))

    for s in schedule_routes:
        thread_el = ElementTree.Element(
            'thread',
            uid=s.thread.uid,
            number=clean_number(s.thread.number),
            title=s.thread.L_title(),
            is_combined="1" if s.thread.is_combined else "0")

        if export_version != '1':
            thread_el.attrib['departure_platform'] = s.rtstation.L_platform()

        if date_ and export_version != '1':
            add_facilities_data(thread_el, s.suburban_facilities)

        if s.update:
            thread_el.attrib['update'] = 'update'

        thread_el.attrib['stops'] = s.schedule.L_stops() or get_default_stops(export_version)
        thread_tz_today = now_aware.astimezone(s.thread.pytz).date()

        old_days_format = request.GET.get('old_days_format')
        timezone_str = request.GET.get('timezone')
        timezone = pytz.timezone(timezone_str) if timezone_str else None

        shift = s.rtstation.calc_days_shift(event=s.event, start_date=s.start_date, event_tz=timezone)
        fill_days_and_except_texts(thread_tz_today, thread_el, s.thread, shift, next_plan, old_days_format=old_days_format)
        cut_days_text_and_stops_text(thread_el)

        direction_title = direction_title_by_code.get(s.schedule_item.direction_code)
        if direction_title:
            thread_el.attrib['direction'] = direction_title

        fill_transport_type(thread_el, s.thread, request.GET.get('add_subtypes'))

        if s.is_last_station:
            thread_el.attrib['terminal'] = 'terminal'

        if s.loc_arrival_datetime:
            arrival = s.loc_arrival_datetime.astimezone(timezone) if timezone else s.loc_arrival_datetime
            thread_el.attrib['arrival'] = arrival.strftime("%H:%M")

        if s.loc_departure_datetime:
            departure = s.loc_departure_datetime.astimezone(timezone) if timezone else s.loc_departure_datetime
            thread_el.attrib['departure'] = departure.strftime("%H:%M")
            if export_sub_version >= 2.1 and date_:
                thread_el.set('departure_utc', s.loc_departure_datetime.astimezone(pytz.utc).isoformat(' '))

        if add_delays:
            debug_delays = request.GET.get('debug_delays')
            add_departure_delays_data(thread_el, s.z_tablo, debug_delays, s.thread.uid)

        station_el.append(thread_el)

    national_version, lang = request.national_version, request.language_code
    teaser_set = TeaserSetExport('tablo', (station, schedule_routes), national_version=national_version, lang=lang)

    return station_el, teaser_set


@xmlresponse
def station_schedule(request, esr_code):
    return station_schedule_base(request, esr_code)[0]


@xmlresponse
def station_schedule_v2(request, esr_code):
    version = '2'
    export_sub_version = get_sub_version(request)
    station_el, teaser_set = station_schedule_base(request, esr_code, add_delays=True,
                                                   version=version, sub_version=export_sub_version)
    teasers_el = teasers_to_xml(teaser_set)

    response_el = ElementTree.Element('response')
    response_el.append(teasers_el)
    response_el.append(station_el)
    if export_sub_version >= 2.1:
        add_server_time(response_el)
    return response_el


def thread_base(request, uid, **kwargs):
    add_delays = kwargs.pop('add_delays', False)
    export_version = kwargs.pop('version', '1')

    thread = get_object_or_404(RThread, uid=uid)
    RouteLTitle.fetch([thread.L_title])

    msk_today = environment.today()
    thread_start_date = thread.get_thread_tz_start_date(msk_today, MSK_TZ)
    naive_start_dt = datetime.combine(thread_start_date, thread.tz_start_time)
    aware_start_dt = thread.pytz.localize(naive_start_dt)

    rtstations = list(thread.path.select_related('station'))
    first_rtstation = rtstations[0]
    first_station = first_rtstation.station

    msk_departure = aware_start_dt.astimezone(MSK_TZ)
    local_departure = aware_start_dt.astimezone(first_station.pytz)

    thread_el = ElementTree.Element(
        'thread',
        title=thread.L_title(),
        number=clean_number(thread.number),
        start_time=local_departure.strftime("%H:%M"),
        start_time_msk=msk_departure.strftime("%H:%M"),
        is_combined="1" if thread.is_combined else "0",
    )

    if thread.update:
        thread_el.attrib['update'] = 'update'

    if thread.cancel:
        thread_el.attrib['cancel'] = 'cancel'

    fill_transport_type(thread_el, thread, request.GET.get('add_subtypes'))

    old_days_format = request.GET.get('old_days_format')
    shift = first_rtstation.calc_days_shift(event='departure', start_date=thread_start_date)
    current_plan, next_plan = TrainSchedulePlan.add_to_threads([thread], environment.today())
    fill_days_and_except_texts(
        thread_start_date, thread_el, thread, shift,
        next_plan=next_plan,
        old_days_format=old_days_format
    )

    try:
        schedule = StationSchedule.get(thread=thread, station=first_station, event='departure')
        thread_el.attrib['stops'] = schedule.L_stops() or get_default_stops(export_version)
    except StationSchedule.DoesNotExist:
        thread_el.attrib['stops'] = u''

    cut_days_text_and_stops_text(thread_el)

    for rtstation in rtstations:
        station = rtstation.station

        title = station.L_title()
        popular_title = station.L_popular_title(fallback=False)
        title = append_station_title_prefix(station, title)

        rtstation_el = ElementTree.Element(
            'rtstation',
            title=title or u'',
            popular_title=popular_title or u'',
            platform=rtstation.L_platform() or "",
            is_combined="1" if rtstation.is_combined else "0",
        )

        esr_code = rtstation.station.get_code('esr')
        if esr_code:
            rtstation_el.attrib['esr'] = esr_code

        for event in ['arrival', 'departure']:
            t = getattr(rtstation, 'tz_%s' % event)
            if t is not None:
                local_dt = rtstation.get_event_loc_dt(event, naive_start_dt)

                rtstation_el.attrib[event] = str(int((local_dt - aware_start_dt).total_seconds() / 60))
                rtstation_el.attrib[event + '_local'] = local_dt.strftime("%H:%M")

        if add_delays:
            if request.GET.get('debug_delays'):
                add_departure_delays_data(rtstation_el, None, debug=True, thread_uid=thread.uid)
            else:
                local_departure_dt = rtstation.get_departure_loc_dt(naive_start_dt)
                if local_departure_dt:
                    try:
                        ztablo = ZTablo2.objects.get(
                            rtstation=rtstation,
                            original_departure=local_departure_dt.replace(tzinfo=None)
                        )
                    except ZTablo2.DoesNotExist:
                        pass
                    else:
                        add_departure_delays_data(rtstation_el, ztablo)

        thread_el.append(rtstation_el)

    national_version, lang = request.national_version, request.language_code
    teaser_set = TeaserSetExport('thread_{}'.format(thread.t_type.code),
                                 (thread, thread.company),
                                 national_version=national_version, lang=lang)

    return thread_el, teaser_set


@xmlresponse
def thread(request, uid):
    return thread_base(request, uid)[0]


@xmlresponse
def thread_v2(request, uid):
    thread_el, teaser_set = thread_base(request, uid, add_delays=True, version='2')

    teasers_el = teasers_to_xml(teaser_set)

    response_el = ElementTree.Element('response')
    response_el.append(teasers_el)
    response_el.append(thread_el)

    return response_el


@xmlresponse
def trip(request, esr_from, esr_to):
    station_from = get_station_by_esr_or_404(esr_from, hidden=True)
    station_to = get_station_by_esr_or_404(esr_to, hidden=True)

    return common_search(request, station_from, station_to)[0]


@xmlresponse
def trip_v2(request, esr_from, esr_to):
    station_from = get_station_by_esr_or_404(esr_from, hidden=True)
    station_to = get_station_by_esr_or_404(esr_to, hidden=True)

    version = '2'
    export_sub_version = get_sub_version(request)
    trip_el, teaser_set = common_search(request, station_from, station_to, add_delays=True,
                                        version=version, sub_version=export_sub_version)
    teasers_el = teasers_to_xml(teaser_set)

    response_el = ElementTree.Element('response')
    response_el.append(teasers_el)
    response_el.append(trip_el)
    if export_sub_version >= 2.1:
        add_server_time(response_el)
    return response_el


def search_base(request, **kwargs):
    station_from_id = esr_to_id(request.GET.get('station_from'), hidden=True)
    station_to_id = esr_to_id(request.GET.get('station_to'), hidden=True)

    city_from_id = request.GET.get('city_from')
    city_to_id = request.GET.get('city_to')

    point_from = get_point(station_from_id, city_from_id)
    point_to = get_point(station_to_id, city_to_id)

    if point_from is None or point_to is None:
        return HttpResponse(u'<error text="unknown city or station" />', content_type='application/xml')

    point_list_from = PointList(point_from, [point_from], None, exact_variant=True)
    point_list_to = PointList(point_to, [point_to], None, exact_variant=True)

    try:
        point_from, point_to = process_points_lists(point_list_from, point_list_to, suburban=True)
    except SamePointError:
        return HttpResponse(u'<error text="the cities or stations are identical" />', content_type='application/xml')

    return common_search(request, point_from.point, point_to.point, **kwargs)


@xmlresponse
def search(request):
    response = search_base(request)
    if isinstance(response, HttpResponse):
        return response

    return response[0]


@xmlresponse
def search_v2(request):
    version = '2'
    export_sub_version = get_sub_version(request)
    response = search_base(request, add_delays=True, version=version, sub_version=export_sub_version)
    if isinstance(response, HttpResponse):
        return response

    trip_el, teaser_set = response

    teasers_el = teasers_to_xml(teaser_set)

    response_el = ElementTree.Element('response')
    response_el.append(teasers_el)
    response_el.append(trip_el)

    if export_sub_version >= 2.1:
        add_server_time(response_el)
    return response_el


def log_search(request, point_from, point_to, when, segments):
    try:
        suburban_code = TransportType.objects.get(id=TransportType.SUBURBAN_ID).code
        t_type_counts = {suburban_code: len(segments)}

        user_agent = request.META.get('HTTP_USER_AGENT')
        user_type = None
        if user_agent:
            if user_agent.startswith('yandex_trains_ios'):
                user_type = 'ios_app'
            elif user_agent.startswith('Android app'):  # example: Android app: 3.20(320)
                user_type = user_agent

        # Нас интересуют только поиски из приложений
        if not user_type:
            return

        when_str = when.strftime("%Y-%m-%d") if when else None

        log_data = {
            'from_id': point_from.point_key,
            'to_id': point_to.point_key,
            'transport_type': suburban_code,
            'when': when_str,
            't_type_counts': json.dumps(t_type_counts, separators=(',', ':')),
            'national_version': getattr(request, 'NATIONAL_VERSION', None),
            'service': 'export',
            'user_type': user_type,
            'tskv_format': 'rasp-users-search-log',
            'referer': request.META.get('HTTP_REFERER'),
            'device_uuid': request.GET.get('uuid')
        }

        users_search_log.log(request, log_data)
    except Exception:
        log.exception('Error when writing to user-search-log')


def common_search(request, point_from, point_to, **kwargs):
    add_delays = kwargs.pop('add_delays', False)
    export_version = kwargs.pop('version', '1')
    export_sub_version = kwargs.get('sub_version', 1)

    if point_from == point_to:
        return ElementTree.Element('trip'), None

    today = environment.today()

    date_str = request.GET.get('date')
    from_tz = pytz.timezone(point_from.time_zone)
    date_ = get_today_tomorrow_date(date_str, from_tz)

    if not date_:
        try:
            date_ = parse_date(date_str)
            if date_:
                if date_.year < 1900:
                    date_ = date_.replace(year=today.year)
                date_.strftime("%Y-%m-%d")
        except (ValueError, TypeError):
            date_ = None

    if date_:
        start = point_from.localize(loc=datetime.combine(date_, time(0, 0)))

        try:
            days_ahead = int(request.GET['days_ahead'])
        except (KeyError, ValueError):
            days_ahead = 1

        if days_ahead < 0:
            days_ahead = 0

        try:
            tomorrow_upto = int(request.GET['tomorrow_upto'])
        except (ValueError, KeyError):
            tomorrow_upto = 0

        if tomorrow_upto > 12:
            tomorrow_upto = 12

        if tomorrow_upto < 0:
            tomorrow_upto = 0

        end = start + timedelta(days=days_ahead, hours=tomorrow_upto)

        segments = []
        for segment in find_suburban(point_from, point_to, start.date()):
            if segment.departure < start:
                continue

            if segment.departure > end:
                break

            segments.append(segment)

        # Костыль для RASP-8365
        border = datetime.combine(date_, time(0, 0))
        segments = [s for s in segments if datetime.replace(s.departure, tzinfo=None) >= border
                    and datetime.replace(s.arrival, tzinfo=None) >= border]
    else:
        segments, _, _ = search_routes(
            point_from, point_to, None,
            [TransportType.objects.get(code='suburban')]
        )

    trip_el = ElementTree.Element('trip')

    tariffs_by_stations = add_suburban_tariffs(segments)

    need_all_tariffs = request.GET.get('all_tariffs', '').lower() == 'true'
    if need_all_tariffs:
        tariffs_el = build_tariffs_by_stations(tariffs_by_stations)
        trip_el.append(tariffs_el)

    if segments:
        trip_el.attrib.update({
            'from': segments[0].station_from.get_code('esr'),
            'to': segments[0].station_to.get_code('esr'),
        })

    if date_:
        trip_el.attrib['date'] = date_.strftime("%Y-%m-%d")
        trip_el.attrib['days_ahead'] = str(days_ahead)
        if export_sub_version >= 2.1:
            day_start_utc_iso, day_end_utc_iso = get_day_start_end_utc_in_iso(date_, point_from.pytz)
            trip_el.set('day_start_utc', day_start_utc_iso)
            trip_el.set('day_end_utc', day_end_utc_iso)

    if add_delays:
        add_z_tablos_to_segments(segments, separated_segments=True)

    RouteLTitle.fetch([segment.thread.L_title for segment in segments])

    station_to_esr = get_station_to_esr(segments)

    current_plan, next_plan = TrainSchedulePlan.add_to_threads([s.thread for s in segments if s.thread], today)

    rtstations, threads = [], []
    for segment in segments:
        rtstations.append(segment.rtstation_from)

        # Предустанавливаем известные объекты, чтобы исключить лишние хождения в базу
        segment.rtstation_from.thread = segment.thread
        threads.append(segment.thread)

    fetch_related(rtstations, 'station', model=RTStation)
    fetch_related(threads, 't_subtype', model=RThread)

    if settings.USERS_SEARCH_LOG:
        log_search(request, point_from, point_to, date_, segments)

    if date_ and export_version != '1':
        fill_segments_suburban_facilities(segments)

    timezone = request.GET.get('timezone')
    build_segment_el_part = partial(
        build_segment_el,
        today=today,
        next_plan=next_plan,
        station_to_esr=station_to_esr,
        add_delays=add_delays,
        delays_debug=request.GET.get('debug_delays'),
        add_subtype=request.GET.get('add_subtypes'),
        old_days_format=request.GET.get('old_days_format'),
        add_days_mask=request.GET.get('add_days_mask'),
        timezone=pytz.timezone(timezone) if timezone else None,
        version=export_version,
        export_sub_version=export_sub_version,
        request_date=date_,
    )

    def sort_segments(time_format):
        segments.sort(key=lambda _s: (_s.departure.strftime(time_format), _s.thread.uid))

    if date_ and days_ahead > 1:
        time_format = '%Y-%m-%d %H:%M'
        sort_segments(time_format)

        for day, segments in groupby(segments, lambda _s: _s.departure.date()):
            day_el = ElementTree.Element('day', date=day.strftime("%Y-%m-%d"))
            for segment in segments:
                day_el.append(build_segment_el_part(segment, time_format))
            trip_el.append(day_el)
    else:
        # При поиске на все дни используем сокращенный формат времени отправления,
        # в том числе для сортировки по departure
        time_format = '%Y-%m-%d %H:%M' if date_ else '%H:%M'
        sort_segments(time_format)

        for segment in segments:
            trip_el.append(build_segment_el_part(segment, time_format))

    national_version, lang = request.national_version, request.language_code
    teaser_set = TeaserSetExport(
        'search_suburban',
        {
            'points': [point_from, point_to],
            'routes': segments
        },
        national_version=national_version,
        lang=lang
    )

    return trip_el, teaser_set


def add_schedule(segments):
    rtstations = [segment.rtstation_from for segment in segments if segment.rtstation_from]

    schedules = dict((s.rtstation_id, s) for s in StationSchedule.objects.filter(rtstation__in=rtstations))

    for segment in segments:
        thread_schedule = schedules.get(segment.rtstation_from.id)

        if thread_schedule:
            thread_schedule.thread = segment.thread

        segment.schedule = thread_schedule


def find_suburban(point_from, point_to, local_date_from):
    return find(point_from, point_to, local_date_from, TransportType.objects.get(code='suburban'))


def build_segment_el(segment, time_format, today, next_plan, **kwargs):
    station_to_esr = kwargs.pop('station_to_esr', {})
    add_delays = kwargs.pop('add_delays', False)
    delays_debug = kwargs.pop('delays_debug', False)
    add_subtype = kwargs.pop('add_subtype', False)
    old_days_format = kwargs.pop('old_days_format', False)
    add_days_mask = kwargs.pop('add_days_mask', False)
    timezone = kwargs.pop('timezone', None)
    export_version = kwargs.pop('version', '1')
    request_date = kwargs.pop('request_date', None)
    export_sub_version = kwargs.pop('export_sub_version', 2)

    assert not kwargs

    if timezone:
        departure = segment.departure.astimezone(timezone)
        arrival = segment.arrival.astimezone(timezone)
    else:
        departure = segment.departure
        arrival = segment.arrival

    el = ElementTree.Element(
        'segment',
        departure=departure.strftime(time_format),
        arrival=arrival.strftime(time_format),
        duration=str(int((segment.arrival - segment.departure).total_seconds() / 60)),
        uid=segment.thread.uid,
        number=clean_number(segment.thread.number),
        title=segment.thread.L_title(),
        title_short=segment.thread.L_title(short=True),
        departure_platform=segment.rtstation_from.L_platform() or "",
        arrival_platform=segment.rtstation_to.L_platform() or "")

    el.set('from', station_to_esr.get(segment.station_from, u''))
    el.set('to', station_to_esr.get(segment.station_to, u''))

    set_segment_tariff(el, segment)

    shift = segment.rtstation_from.calc_days_shift(
        event='departure',
        start_date=segment.start_date,
        event_tz=timezone)

    fill_days_and_except_texts(
        today, el, segment.thread, shift, next_plan,
        old_days_format=old_days_format,
        add_days_mask=add_days_mask)

    if segment.thread.update:
        el.attrib['update'] = 'update'

    if segment.thread.cancel:
        el.attrib['cancel'] = 'cancel'

    el.attrib['stops'] = segment.L_stops() or get_default_stops(export_version)

    if export_version != '1':
        add_facilities_data(el, segment.suburban_facilities)

    if export_sub_version >= 2.1 and request_date:
        el.set('departure_utc', departure.astimezone(pytz.utc).isoformat(' '))

    cut_days_text_and_stops_text(el)

    fill_transport_type(el, segment.thread, add_subtype)

    if add_delays:
        add_departure_delays_data(el, segment.departure_z_tablo, delays_debug, segment.thread.uid)

    return el


def fill_days_and_except_texts(today, el, thread, shift, next_plan=None, **kwargs):
    if kwargs.pop('old_days_format', False):
        return fill_days_and_except_texts_old(today, el, thread, shift, next_plan)

    add_days_mask = kwargs.pop('add_days_mask', False)

    days_data = thread.L_days_text_dict(
        shift=shift,
        thread_start_date=today,
        next_plan=next_plan,
        show_days=True)

    el.attrib['days'] = days_data['days_text']
    except_text = days_data.get('except_days_text')
    if except_text:
        el.attrib['except'] = except_text

    if add_days_mask:
        mask = str(RunMask(thread.year_days, today).shifted(shift))
        schedule_el = ElementTree.Element('schedule')
        for month in range(1, 13):
            days = mask[(month - 1) * 31: month * 31]
            year = today.year if today.month <= month else today.year + 1
            schedule_el.append(ElementTree.Element(
                'mask',
                year=str(year), month=str(month), days=days
            ))
        el.append(schedule_el)


def fill_days_and_except_texts_old(today, el, thread, shift, next_plan=None):
    """
    https://st.yandex-team.ru/RASPEXPORT-88
    Старый формат, сохраненный для Фетисова.
    """
    if thread.update or thread.assignment:
        text = u", ".join(d.strftime("%d.%m")
                          for d in RunMask(thread.year_days, today=today).shifted(shift).dates())

    else:
        text = thread.plain_days_text(shift, lang=get_language())

        if text and thread.schedule_plan:
            appendix = thread.schedule_plan.get_L_appendix(today, next_plan)

            if appendix:
                text += u', %s' % appendix

        if not text:
            text = u", ".join(d.strftime("%d.%m")
                              for d in RunMask(thread.year_days, today=today).shifted(shift).dates()[:10])

    el.attrib['days'] = text

    except_text = thread.except_days_text(shift)

    if except_text:
        el.attrib['except'] = except_text


def cut_days_text_and_stops_text(xml_el):
    """ Ограничение длины по RASP-8798 """

    if xml_el.attrib.get('days'):
        xml_el.attrib['days'] = xml_el.attrib['days'][:1000]

    if xml_el.attrib.get('stops'):
        xml_el.attrib['stops'] = xml_el.attrib['stops'][:1000]


def fill_transport_type(el, thread, add_subtype=False):
    if thread.express_type == 'aeroexpress':
        el.attrib['type'] = thread.express_type
    elif thread.express_type == 'express':
        el.attrib['type'] = thread.express_type

    if add_subtype and thread.t_subtype and thread.t_subtype_id != TransportSubtype.SUBURBAN_ID:
        color = thread.t_subtype.color.color if thread.t_subtype.color else ''

        subtype_el = ElementTree.Element(
            'transport_subtype',
            code=thread.t_subtype.code,
            title=thread.t_subtype.L_title(),
            color=color,
        )
        el.append(subtype_el)


def clean_number(number):
    return re.sub(ur'[^0-9/]+', u'', number, flags=re.U)


def get_station_to_esr(segments):
    station_set = set()

    for s in segments:
        station_set.add(s.station_from)
        station_set.add(s.station_to)

    return {s: s.get_code('esr') for s in station_set}


def esr_to_id(esr_code, hidden=False):
    try:
        return get_station_by_esr_or_404(esr_code, hidden).id

    except Http404:
        return None


def get_point(station_id, city_id):
    if station_id:
        try:
            return Station.objects.get(id=station_id)
        except Station.DoesNotExist:
            pass

    if city_id:
        try:
            return Settlement.objects.get(id=city_id)
        except Settlement.DoesNotExist:
            pass

    return None


def get_station_by_esr_or_404(esr_code, hidden=False):
    try:
        if hidden:
            return Station.hidden_manager.get(code_set__system__code__iexact='esr',
                                              code_set__code__iexact=esr_code)
        else:
            return Station.get_by_code('esr', esr_code)

    except Station.DoesNotExist:
        raise Http404()
