# coding: utf8

import logging
from collections import defaultdict
from datetime import timedelta, datetime
from django.db.models import Q, Prefetch
from itertools import chain

from common.models.geo import StationCode, CodeSystem, ExternalDirection, Settlement
from common.models.schedule import RThread, RTStation
from travel.rasp.library.python.common23.date import environment
from common.utils.date import MSK_TZ, RunMask, parse_date
from common.utils.httpresponses import jsonp_response
from common.views.teasers import TeaserSetRaspBase

from stationschedule import SUBURBAN, get_schedule_class

from travel.rasp.export.direction_stations import direction_stations_by_geoid
from travel.rasp.export.export.views.in_xml import get_station_by_esr_or_404
from travel.rasp.export.export.views.utils import append_station_title_prefix


logger = logging.getLogger(__name__)


def get_next_date(week_day_number):
    """
    Возвращает ближайшую дату по индексу дня недели.
    Текущий день не учитывается.
    Понедельник - 0.
    """
    today = environment.today()
    today_number = today.weekday()
    date = today + timedelta(days=week_day_number - today_number)
    return date if today_number < week_day_number else date + timedelta(weeks=1)


def gen_thread_info(thread_object, msk_today, station_esr_codes):
    thread_start_date = thread_object.get_thread_tz_start_date(msk_today, MSK_TZ)
    naive_start_dt = datetime.combine(thread_start_date, thread_object.tz_start_time)
    aware_start_dt = thread_object.pytz.localize(naive_start_dt)

    thread = {'stations': [],
              'uid': thread_object.uid,
              'number': thread_object.number,
              'title': thread_object.L_title()}

    rtstations = list(thread_object.rtstation_set.all())
    departure_local_dt = rtstations[0].get_event_loc_dt('departure', naive_start_dt)
    thread['departure_from'] = departure_local_dt.strftime("%H:%M")
    arrival_local_dt = rtstations[-1].get_event_loc_dt('arrival', naive_start_dt)
    thread['arrival_to'] = arrival_local_dt.strftime("%H:%M")

    for rtstation in rtstations:
        station_object = rtstation.station
        station = {'esr': station_esr_codes.get(station_object.id),
                   'title': append_station_title_prefix(station_object, station_object.L_title())}

        for event in ['arrival', 'departure']:
            local_dt = rtstation.get_event_loc_dt(event, naive_start_dt)
            station[event] = str(int((local_dt - aware_start_dt).total_seconds() / 60)) if local_dt else None

        thread['stations'].append(station)
    return thread


def gen_threads_info(ids_by_directions):
    all_ids, basic_ids = [], []
    for direction, ids_by_dir in ids_by_directions.items():
        for dir_name, ids_by_dates in ids_by_dir.items():
            for day, ids in ids_by_dates.items():
                all_ids.extend(chain(*ids.values()))
                basic_ids.extend(ids['basic'])

    msk_today = environment.today()
    threads = {}
    # Получаем объекты всех полученных ниток и ниток изменений.
    # Получаем связанные rtstation и station.
    prefetch_rtstations = Prefetch('rtstation_set', queryset=RTStation.objects.order_by('id').select_related('station'))
    thread_objects = (RThread.objects.filter(Q(id__in=all_ids) | Q(basic_thread_id__in=basic_ids)).
                      prefetch_related(prefetch_rtstations))

    # Получаем esr коды всех станций всех ниток.
    station_esr_codes = StationCode.StationCodeByStationIdGetter(
        CodeSystem.objects.get(code='esr'),
        station_ids=[rtstation.station.id for thread_object in thread_objects for rtstation in thread_object.rtstation_set.all()]
    )

    # Заполняем информацию по каждой нитке.
    for thread_object in thread_objects:
        threads[thread_object.id] = gen_thread_info(thread_object, msk_today, station_esr_codes)
    return threads


def get_thread_changes(thread_id, date):
    today = environment.today()
    linked_threads_objs = RThread.objects.filter(basic_thread_id=thread_id).prefetch_related('type')

    # Получаем информацию об изменении нитки на нужную дату.
    for thread in linked_threads_objs:
        first_rtstation = list(thread.path.select_related('station'))[0]
        thread_start_date = thread.get_thread_tz_start_date(today, MSK_TZ)
        shift = first_rtstation.calc_days_shift(event='departure', start_date=thread_start_date)
        mask = RunMask(thread.year_days, today).shifted(shift)
        if RunMask.runs_at(str(mask), date):
            return {'change_uid': thread.uid,
                    'change_id': thread.id,
                    'change_type': thread.type.code,
                    'change_title': thread.L_title()}


def stops_diff(base_thread, changed_thread):
    canceled, added = [], []
    # Получаем названия отмененных и назначенных остановок.
    if len(base_thread['stations']) > 2 and len(changed_thread['stations']) > 2:
        for base_station in base_thread['stations']:
            for changed_station in changed_thread['stations']:
                if changed_station['esr'] == base_station['esr']:
                    if (base_station['arrival'] != base_station['departure'] and
                            changed_station['arrival'] == changed_station['departure']):
                        canceled.append(changed_station['title'])

        for changed_station in changed_thread['stations']:
            for base_station in base_thread['stations']:
                if changed_station['esr'] == base_station['esr']:
                    if (base_station['arrival'] == base_station['departure'] and
                            changed_station['arrival'] != changed_station['departure']):
                        added.append(changed_station['title'])
    return canceled, added


def get_schedule(esr_code):
    now_aware = environment.now_aware()
    station = get_station_by_esr_or_404(esr_code)
    result_schedule = {}

    schedule_cls = get_schedule_class(station, schedule_type=SUBURBAN, t_type_code=SUBURBAN)
    schedule = schedule_cls(station, requested_direction='all')
    schedule.build()
    schedule_routes = schedule.schedule
    schedule.fill_directions(schedule_routes)
    direction_title_by_code = schedule.direction_title_by_code

    for schedule_route in schedule_routes:
        thread = schedule_route.thread
        thread_start_date = thread.get_thread_tz_start_date(now_aware.date(), MSK_TZ)
        shift = thread.path[0].calc_days_shift(event='departure', start_date=thread_start_date)

        thread_attrs = {'id':  thread.id,
                        'uid': thread.uid,
                        'type': thread.type.code,
                        'direction': direction_title_by_code.get(schedule_route.schedule_item.direction_code)}
        if thread_attrs['type'] in ['change', 'assignment']:
            thread_attrs['days'] = RunMask(thread.year_days, today=thread_start_date).shifted(shift).dates()
        if thread_attrs['type'] == 'basic':
            thread_attrs['except'] = thread.except_days_text(shift)

        result_schedule[thread_attrs['id']] = thread_attrs
    return result_schedule


def get_direction_threads(esr_code, direction, dates):
    schedule = get_schedule(esr_code)
    ids = {date: defaultdict(list) for date in dates}

    # Группируем нитки, проходящие через станцию, по типу и дате.
    for date in dates:
        d_m_date = date.strftime('%d.%m')
        for thread_id, schedule_item in schedule.items():
            if schedule_item['direction'] == direction:
                if schedule_item['type'] == 'assignment' and date in schedule_item['days']:
                    ids[date]['assignment'].append(thread_id)
                elif schedule_item['type'] == 'basic' and d_m_date in schedule_item['except']:
                    ids[date]['basic'].append(thread_id)
    return ids


def generate_direction_changes(threads_info, ids, date):
    changes = []

    for basic_id in ids['basic']:
        # Получаем тип изменения нитки.
        # Заполняем изменение в зависимости от типа.
        change_thread_info = get_thread_changes(basic_id, date)
        if not change_thread_info:
            logger.exception('Basic_thread_without_changes_id: %s %s', str(basic_id), strdate(date))
            continue
        bs = threads_info[basic_id]
        change = {
            'type': change_thread_info['change_type'],
            'number': bs['number'],
            'base_departure': bs['departure_from'],
            'base_arrival': bs['arrival_to'],
            'base_title': bs['title']}
        if change_thread_info['change_type'] == 'cancel':
            change.update({'subtext': u'Отменен',
                           'color': 'red'})
            changes.append(change)
        elif change_thread_info['change_type'] == 'change':
            ch = threads_info[change_thread_info['change_id']]
            canceled, added = stops_diff(bs, ch)
            change.update({'departure': ch['departure_from'],
                           'arrival': ch['arrival_to']})

            if bs['title'] == ch['title']:
                change['subtype'] = 'timetable'
                change['subtext'] = u'Поезд проследует изменённым расписанием'
            else:
                change['title'] = ch['title']
                if len(bs['stations']) > len(ch['stations']):
                    change['subtype'] = 'reduceroute'
                    change['subtext'] = u'Поезд проследует укороченным маршрутом'

                else:
                    change['subtype'] = 'changeroute'
                    change['subtext'] = u'Поезд проследует изменённым маршрутом'

            if canceled:
                change['stops_text'] = u'Отменены остановки: {}.'.format(u', '.join(canceled))
            if added:
                added_stops = u'Назначены остановки: {}.'.format(u', '.join(added))
                if change.get('stops_text', None):
                    change['stops_text'] += u' {}'.format(added_stops)
                else:
                    change['stops_text'] = added_stops
            changes.append(change)

    for assignment_id in ids['assignment']:
        bs = threads_info[assignment_id]
        thread_type = 'assignment'
        changes.append({
            'type': thread_type,
            'number': bs['number'],
            'departure': bs['departure_from'],
            'arrival': bs['arrival_to'],
            'title': bs['title'],
            'subtype': thread_type,
            'subtext': u'Назначается',
            'color': 'navy',
        })

    return changes


def strdate(date):
    return date.strftime('%Y-%m-%d')


@jsonp_response
def schedule_changes(request):
    geo_id = int(request.GET.get('geo_id'))
    start_date = parse_date(request.GET.get('start_date'))
    end_date = parse_date(request.GET.get('end_date'))
    settlement = Settlement.objects.get(_geo_id=geo_id)
    dates = [start_date + timedelta(days=i) for i in range((end_date - start_date).days + 1)]

    # Получаем id ниток, проходящих через станции всех нужных направлений.
    ids_by_directions = defaultdict(lambda: defaultdict(dict))
    stations = direction_stations_by_geoid.get(geo_id)
    for direction_title, sts_dirs in stations.items():
        direction_object = ExternalDirection.objects.get(suburban_zone=settlement.suburban_zone, full_title=direction_title)

        for esr, sub_dir, dir_name in [(sts_dirs[0], sts_dirs[2], 'forward'),
                                       (sts_dirs[5], sts_dirs[7], 'backward')]:
            ids_by_directions[direction_object][dir_name].update(get_direction_threads(esr, sub_dir, dates))

    # Собираем необходимую информацию по каждой нитке.
    threads_info = gen_threads_info(ids_by_directions)

    directions = []

    # Заполняем информацию об изменениях для каждой нитки.
    for direction_object, ids_by_dir in ids_by_directions.items():
        direction = {'name': direction_object.L_full_title(),
                     'id': direction_object.id,
                     'forward': [], 'backward': []}

        # Добавляем текст тизеров.
        # Класс TeaserSetExport переопределяет filter_hard_params,
        # чтобы игнорировать тизеры с пустым mobile_content.
        # Поэтому используем TeaserSetRaspBase.
        teaser_set = TeaserSetRaspBase('export', 'direction', data=direction_object)
        teasers = teaser_set.get_teasers('direction')
        if teasers:
            direction['texts'] = [teaser.content for teaser in teasers if teaser.content]

        for dir_name, ids_by_date in ids_by_dir.items():
            for date, ids in ids_by_date.items():
                date_changes = {'date': strdate(date)}
                changes = generate_direction_changes(threads_info, ids, date)
                if changes:
                    date_changes['changes'] = changes
                direction[dir_name].append(date_changes)

        directions.append(direction)

    return {'geo_id': geo_id,
            'name': settlement.L_title(),
            'directions': directions}
