# -*- coding: utf-8 -*-

from collections import defaultdict

from xml.etree import ElementTree
from django.shortcuts import get_object_or_404

from common.models.geo import Station, SuburbanZone
from common.models.transport import TransportType
from common.utils.httpcaching import modified_by_timestamp_cached
from common.utils.httpresponses import xmlresponse
from travel.rasp.export.export.views.city import export_stations_list


CACHE_TIMEOUT = 2 * 7 * 24 * 3600  # 2 weeks


@modified_by_timestamp_cached('prepare_all', timeout=CACHE_TIMEOUT)
@xmlresponse
def zones(request):
    u"""Список пригородных зон"""
    tree = ElementTree.Element('suburban_zones')
    for zone in SuburbanZone.objects.all().select_related('settlement'):
        zone_el = ElementTree.Element(
            'zone',
            id=str(zone.id),
            title=zone.L_title() or '',
            code=zone.code or '',
            settlement_id=str(zone.settlement.id),
            settlement_title=zone.settlement.L_title() or '')

        for settlement in zone.settlement_set.filter(hidden=False):
            if settlement.latitude is None or settlement.longitude is None:
                continue

            settlement_el = ElementTree.Element(
                'settlement',
                id=str(settlement.id),
                title=settlement.L_title(),
                longitude=str(settlement.longitude or ""),
                latitude=str(settlement.latitude or ""),
                geo_id=str(settlement._geo_id or ""))
            zone_el.append(settlement_el)

        tree.append(zone_el)

    return tree


@modified_by_timestamp_cached('prepare_all', timeout=CACHE_TIMEOUT)
@xmlresponse
def zones_v2(request):
    u"""Список пригородных зон"""
    tree = ElementTree.Element('suburban_zones')
    for zone in SuburbanZone.objects.all().select_related('settlement'):
        zone_el = ElementTree.Element(
            'zone',
            id=str(zone.id),
            title=zone.L_title() or '',
            code=zone.code or '',
            settlement_id=str(zone.settlement.id),
            settlement_title=zone.settlement.L_title() or ''
        )

        for settlement in zone.settlement_set.filter(hidden=False):
            if settlement.latitude is None or settlement.longitude is None:
                continue

            settlement_el = ElementTree.Element(
                'settlement',
                id=str(settlement.id),
                title=settlement.L_title(),
                longitude=str(settlement.longitude or ''),
                latitude=str(settlement.latitude or ''),
                geo_id=str(settlement._geo_id or ''),
                region=settlement.region.L_title() if settlement.region else '',
                country=settlement.country.L_title() if settlement.country else ''
            )
            zone_el.append(settlement_el)

        tree.append(zone_el)

    return tree


@modified_by_timestamp_cached('prepare_all', timeout=CACHE_TIMEOUT)
@xmlresponse
def zone(request, zone_id):
    zone = get_object_or_404(SuburbanZone, id=zone_id)
    add_station_codes = request.GET.get('add_station_codes', '').lower() == 'true'

    markers = zone.markers()
    stations = Station.objects.filter(id__in=[marker.station_id for marker in markers])

    tree = ElementTree.Element('suburban_zone', id=zone_id)

    export_stations_list(tree, stations, markers, city=zone.settlement, add_station_codes=add_station_codes)

    return tree


@modified_by_timestamp_cached('prepare_all', timeout=CACHE_TIMEOUT)
@xmlresponse
def zone_v2(request, zone_id):
    zone = get_object_or_404(SuburbanZone, id=zone_id)
    add_station_codes = request.GET.get('add_station_codes', '').lower() == 'true'

    markers = zone.markers()
    stations = Station.objects.filter(id__in=[marker.station_id for marker in markers])

    stations_ids_by_title = defaultdict(lambda: defaultdict(list))
    regions_ids = {station.region.id for station in stations if station.region}

    region_stations = Station.objects.filter(region_id__in=regions_ids, t_type_id__in=[TransportType.TRAIN_ID, TransportType.SUBURBAN_ID])
    for station in region_stations:
        stations_ids_by_title[station.region][station.title].append(station.id)

    tree = ElementTree.Element('suburban_zone', id=zone_id)

    export_stations_list(tree, stations, markers, city=zone.settlement, add_station_codes=add_station_codes,
                         stations_ids_by_title=stations_ids_by_title)

    return tree
