package main

import (
	"context"
	stdLog "log"
	"os"
	"sync"

	"a.yandex-team.ru/library/go/core/log"
	"a.yandex-team.ru/travel/library/go/configuration"
	"a.yandex-team.ru/travel/library/go/metrics"
	metricserver "a.yandex-team.ru/travel/library/go/metrics/server"
	"a.yandex-team.ru/travel/library/go/tracing"
	"a.yandex-team.ru/travel/library/go/vault"

	"a.yandex-team.ru/travel/rasp/http_proxy_cache/internal"
	"a.yandex-team.ru/travel/rasp/http_proxy_cache/internal/cache"
	"a.yandex-team.ru/travel/rasp/http_proxy_cache/internal/server"
)

type Config struct {
	ServerConfig *server.ServerConfig `config:"server,required" yaml:"server"`
	RedisConfig  *cache.RedisConfig   `config:"redis,required" yaml:"redis"`
	LogConfig    *internal.LogConfig  `config:"log,required" yaml:"log"`
	Metrics      *metricserver.MetricsConfig
}

const (
	RaspHTTPProxyCacheTesting    = "sec-01dq9hb84rda3gadpkd96bqs9m"
	RaspHTTPProxyCacheProduction = "sec-01drbmfhjfmtjs9rrcdgnsd858"
)

func GetSecUID() string {
	envType := os.Getenv("YENV_TYPE")
	secUID := ""
	switch envType {
	case "production":
		secUID = RaspHTTPProxyCacheProduction
	case "development":
		fallthrough
	case "testing":
		secUID = RaspHTTPProxyCacheTesting
	}
	return secUID
}

func main() {
	tracerCloser := tracing.InitializeDefaultTracer("trains.http_proxy_cache")
	defer tracerCloser.Close()
	loader := configuration.NewDefaultConfitaLoader()
	config := Config{
		Metrics: &metricserver.DefaultMetricsConfig,
	}
	_ = loader.Load(context.Background(), &config)
	logger, err := internal.New(config.LogConfig)
	if err != nil {
		stdLog.Fatalf("can't set up logger: %v", err)
	}
	logger.Info("logger started")

	metricsRegistry := metrics.NewRegistryWithDeployTags()
	appMetricsRegistry := metricsRegistry.WithPrefix("app")
	appMetrics := metrics.NewAppMetrics(appMetricsRegistry)
	metrics.SetGlobalAppMetrics(appMetrics)

	r := vault.NewYavSecretsResolverFromEnv("RASP_VAULT_OAUTH_TOKEN")
	secUID := GetSecUID()
	redisPwd, err := r.GetSecretValue(secUID, "REDIS_HTTP_PROXY_CACHE_PASSWORD")
	if err != nil {
		logger.Fatal("cannot get proxy cache password:", log.Error(err))
	}
	var client server.Cacher = cache.NewRedisClient(config.RedisConfig, redisPwd, logger)

	waitGroup := sync.WaitGroup{}
	waitGroup.Add(2)

	go func() {
		defer waitGroup.Done()
		err = server.RunServer(config.ServerConfig, client, logger, metricsRegistry.WithPrefix("http"))
		if err != nil {
			logger.Error("cannot run server:", log.Error(err))
		}
	}()

	go func() {
		defer waitGroup.Done()

		err = metricserver.RunMetricsHTTPServer(context.Background(), *config.Metrics, logger, metricsRegistry)
		if err != nil {
			logger.Error("cannot run metrics server:", log.Error(err))
		}
	}()

	logger.Info("Started")
	waitGroup.Wait()
}
