package cache

import (
	"context"
	"errors"
	"fmt"
	"time"

	"github.com/go-redis/redis/v8"
	"github.com/golang/protobuf/proto"

	"a.yandex-team.ru/library/go/core/log"
	"a.yandex-team.ru/library/go/core/metrics"
	tm "a.yandex-team.ru/travel/library/go/metrics"
	http_proxy_cache "a.yandex-team.ru/travel/rasp/http_proxy_cache/proto"
)

var CacheMissError error = errors.New("cache miss")

type RedisConfig struct {
	Addrs      []string `yaml:"addrs"`
	MasterName string   `yaml:"master_name"`
}

type RedisCache struct {
	Config      *RedisConfig
	RedisClient redis.UniversalClient
	Logger      log.Logger
	metrics     *cacheMetrics
}

type cacheMetrics struct {
	SetCount      metrics.Counter
	SetErrorCount metrics.Counter
	GetCount      metrics.Counter
	GetHitCount   metrics.Counter
	GetMissCount  metrics.Counter
	GetErrorCount metrics.Counter
}

func (rc *RedisCache) Set(key string, value *http_proxy_cache.TCacheRecord, expiration time.Duration) error {
	rc.metrics.SetCount.Inc()
	v, err := proto.Marshal(value)
	if err != nil {
		rc.metrics.SetErrorCount.Inc()
		return fmt.Errorf("value serialize error: %w", err)
	}
	return rc.RedisClient.Set(context.TODO(), key, v, expiration).Err()
}

func (rc *RedisCache) Get(key string) (*http_proxy_cache.TCacheRecord, error) {
	rc.metrics.GetCount.Inc()
	v, err := rc.RedisClient.Get(context.TODO(), key).Bytes()
	if err == redis.Nil {
		rc.metrics.GetMissCount.Inc()
		rc.Logger.Debugf("cache miss %v", key)
		return nil, CacheMissError
	} else if err != nil {
		rc.metrics.GetErrorCount.Inc()
		return nil, err
	}
	value := &http_proxy_cache.TCacheRecord{}
	err = proto.Unmarshal(v, value)
	if err != nil {
		rc.metrics.GetErrorCount.Inc()
		return nil, fmt.Errorf("value deserialize error: %w", err)
	}
	rc.Logger.Debugf("cache hit %v\n response from cache %v", key, string(value.ResponseBody))
	rc.metrics.GetHitCount.Inc()
	return value, nil
}

func (rc *RedisCache) Del(key string) error {
	_, err := rc.RedisClient.Del(context.TODO(), key).Result()
	return err
}

func NewRedisClient(config *RedisConfig, redisPwd string, l log.Logger) *RedisCache {
	client := redis.NewUniversalClient(
		&redis.UniversalOptions{
			Addrs:      config.Addrs,
			MasterName: config.MasterName,
			Password:   redisPwd,
			DB:         0,
			ReadOnly:   false,
		},
	)
	return &RedisCache{Config: config, RedisClient: client, Logger: l, metrics: newCacheMetrics()}
}

func newCacheMetrics() *cacheMetrics {
	am := tm.GlobalAppMetrics()
	return &cacheMetrics{
		SetCount:      am.GetOrCreateCounter("cache-set", nil, "count"),
		SetErrorCount: am.GetOrCreateCounter("cache-set", nil, "errors_count"),
		GetCount:      am.GetOrCreateCounter("cache-get", nil, "count"),
		GetHitCount:   am.GetOrCreateCounter("cache-get", nil, "hit_count"),
		GetMissCount:  am.GetOrCreateCounter("cache-get", nil, "miss_count"),
		GetErrorCount: am.GetOrCreateCounter("cache-get", nil, "errors_count"),
	}
}
