package server

import (
	"net/http"
	"time"

	"a.yandex-team.ru/library/go/core/log"
	"a.yandex-team.ru/library/go/core/metrics"
	"a.yandex-team.ru/library/go/httputil/middleware/httpmetrics"
	"a.yandex-team.ru/travel/library/go/tracing"
	http_proxy_cache "a.yandex-team.ru/travel/rasp/http_proxy_cache/proto"
	"github.com/go-chi/chi/v5"
	"github.com/go-chi/chi/v5/middleware"
)

type ServerConfig struct {
	Port     string          `yaml:"port"`
	Services []ServiceConfig `yaml:"services"`
}

type Cacher interface {
	Set(key string, value *http_proxy_cache.TCacheRecord, expiration time.Duration) error
	Del(key string) error
	Get(key string) (*http_proxy_cache.TCacheRecord, error)
}

func RunServer(config *ServerConfig, cache Cacher, logger log.Logger, metricsRegistry metrics.Registry) error {
	tm := tracing.NewTracingMiddlewareBuilder().
		WithFilter(tracing.NewPathFilter("/ping")).
		WithExtractor(tracing.NewHeaderTagExtractor("request-id", "X-Request-Id")).
		Build()
	r := chi.NewRouter()
	r.Use(middleware.RequestID)
	r.Use(middleware.Recoverer)

	ping := func(w http.ResponseWriter, r *http.Request) {
		_, err := w.Write([]byte("Pong"))
		if err != nil {
			logger.Error("ping error:", log.Error(err))
		}
	}
	r.Get("/ping", ping)

	rh := r.With(tm.Handle)
	rh.Use(httpmetrics.New(metricsRegistry, httpmetrics.WithPathEndpoint(), httpmetrics.WithSolomonRated()))
	for _, sc := range config.Services {
		service := NewService(&sc, cache, logger)
		rh.Route(sc.BasePath, func(rh chi.Router) {
			rh.Get("/*", service.Get)
		})
	}

	return http.ListenAndServe(":"+config.Port, r)
}
