package server

import (
	"a.yandex-team.ru/library/go/core/log"
	"a.yandex-team.ru/travel/rasp/http_proxy_cache/internal/cache"
	"a.yandex-team.ru/travel/rasp/http_proxy_cache/internal/utils"
	http_proxy_cache "a.yandex-team.ru/travel/rasp/http_proxy_cache/proto"
	"errors"
	"fmt"
	"github.com/golang/protobuf/ptypes"
	"github.com/opentracing/opentracing-go"
	"net/http"
	"strings"
	"time"
)

const CacheFormatVersion = 1

type ServiceConfig struct {
	BasePath       string         `yaml:"base_path"`
	Backend        string         `yaml:"backend"`
	BackendTimeout time.Duration  `yaml:"backend_timeout"`
	Caching        *CachingConfig `yaml:"caching"`
}

type CachingConfig struct {
	IgnoreQueryParams     []string      `yaml:"ignore_query_params"`
	IncludeRequestHeaders []string      `yaml:"include_request_headers"`
	TTL                   time.Duration `yaml:"time_to_live"`
	TimeToRefresh         time.Duration `yaml:"time_to_refresh"`
}

type Service struct {
	Config *ServiceConfig
	Cache  Cacher
	Logger log.Logger
	Client *http.Client
}

func NewService(Config *ServiceConfig, Cache Cacher, Logger log.Logger) Service {
	client := http.Client{Timeout: Config.BackendTimeout}
	var res = Service{Config: Config, Cache: Cache, Logger: Logger, Client: &client}
	return res
}

func (s *Service) getCacheKey(r *http.Request) string {
	query := r.URL.Query()
	for _, qp := range s.Config.Caching.IgnoreQueryParams {
		query.Del(qp)
	}
	var headers []string
	for _, header := range s.Config.Caching.IncludeRequestHeaders {
		v := r.Header.Get(header)
		if v != "" {
			headers = append(headers, fmt.Sprintf("%s=%s", header, v))
		}
	}
	return fmt.Sprintf("v%v:%v:%v?%v", CacheFormatVersion, strings.Join(headers, ";"), r.URL.Path, query.Encode())
}

func (s *Service) writeErrorResponse(w http.ResponseWriter, msg string, err error) {
	s.Logger.Error(msg, log.Error(err))
	w.WriteHeader(http.StatusInternalServerError)
	_, _ = w.Write([]byte(err.Error()))
}

func (s *Service) writeResponse(w http.ResponseWriter, rec *http_proxy_cache.TCacheRecord, now time.Time) error {
	responseBody := rec.ResponseBody
	for _, ph := range rec.ResponseHeader {
		w.Header().Del(ph.Key)
		for _, v := range ph.Values {
			w.Header().Add(ph.Key, v)
		}
	}
	w.Header().Set("Date", now.Format(time.RFC1123))
	w.WriteHeader(int(rec.ResponseStatusCode))
	_, err := w.Write(responseBody)
	return err
}

func (s *Service) getFromBackendAndCacheIfPossible(r *http.Request, cacheKey string) (*http_proxy_cache.TCacheRecord, error) {
	getFromBackendAndCacheIfPossibleSpan, _ := opentracing.StartSpanFromContext(r.Context(), "getFromBackendAndCacheIfPossible")
	defer getFromBackendAndCacheIfPossibleSpan.Finish()

	servicePath := strings.TrimPrefix(r.RequestURI, s.Config.BasePath)
	backendURL := s.Config.Backend + servicePath
	s.Logger.Infof("Query to backend %v", backendURL)
	backendRequest, err := http.NewRequest("GET", backendURL, nil)
	if err != nil {
		return nil, fmt.Errorf("error create request: %w", err)
	}
	backendRequest.Header = r.Header

	err = opentracing.GlobalTracer().Inject(
		getFromBackendAndCacheIfPossibleSpan.Context(),
		opentracing.HTTPHeaders,
		opentracing.HTTPHeadersCarrier(backendRequest.Header))
	if err != nil {
		s.Logger.Error("tracing injection error:", log.Error(err))
	}

	backendResponse, err := s.Client.Do(backendRequest)
	if err != nil {
		return nil, fmt.Errorf("error request %v: %w", backendURL, err)
	}
	rec, err := utils.NewCacheRecord(backendResponse)
	if err != nil {
		return nil, fmt.Errorf("error read response %v: %w", backendURL, err)
	}
	s.Logger.Debugf("response from backend status=%v, data=%v", rec.ResponseStatusCode, string(rec.ResponseBody))
	if rec.ResponseStatusCode >= 200 && rec.ResponseStatusCode < 499 {
		err = s.Cache.Set(cacheKey, rec, s.Config.Caching.TTL)
		if err != nil {
			s.Logger.Error("cache setting error: ", log.Error(err))
		}
	} else {
		err = s.Cache.Del(cacheKey)
		if err != nil {
			s.Logger.Error("cache deleting error: ", log.Error(err))
		}
	}
	return rec, nil
}

func (s *Service) Get(w http.ResponseWriter, r *http.Request) {
	hpcHandlerSpan, _ := opentracing.StartSpanFromContext(r.Context(), "Get")
	defer hpcHandlerSpan.Finish()
	now := time.Now().UTC()
	cacheKey := s.getCacheKey(r)
	rec, err := s.Cache.Get(cacheKey)
	if err == nil {
		err = s.writeResponse(w, rec, now)
		if err != nil {
			s.Logger.Error("cannot write response with data: ", log.Error(err))
		}
		createdAt, err := ptypes.Timestamp(rec.CreatedAt)
		needRefresh := false
		if err != nil {
			s.Logger.Error("cannot convert createdAt from proto: ", log.Error(err))
			needRefresh = true
		} else {
			needRefresh = now.Sub(createdAt) > s.Config.Caching.TimeToRefresh
		}
		if needRefresh {
			s.Logger.Debugf("Need refresh. CacheKey: %v, CachedAt: %v", cacheKey, createdAt)
			go func() {
				_, err := s.getFromBackendAndCacheIfPossible(r, cacheKey)
				if err != nil {
					s.Logger.Error("cannot get from backend and cache: ", log.Error(err))
				}
			}()
		}
		return
	}

	if errors.Is(err, cache.CacheMissError) {
		s.Logger.Debugf("cache miss with key: %v", cacheKey)
	} else {
		s.Logger.Error("cannot get form cache: ", log.Error(err))
	}
	rec, err = s.getFromBackendAndCacheIfPossible(r, cacheKey)
	if err != nil {
		s.writeErrorResponse(w, "get from backend and cache error: ", err)
		return
	}
	err = s.writeResponse(w, rec, now)
	if err != nil {
		s.Logger.Error("cannot write response: ", log.Error(err))
	}
}
