# coding: utf8
from __future__ import unicode_literals, absolute_import, division, print_function

import logging
from datetime import datetime, timedelta

from pymongo.errors import OperationFailure

from common.db import maintenance
from common.models.timestamp import Timestamp
from common.utils.date import MSK_TZ, UTC_TZ
from common.utils.lock import lock
from travel.rasp.library.python.common23.logging.scripts import script_context
from travel.rasp.info_center.info_center.scripts.utils import SUBS_CHANGES_LOCK, SUBS_LOCK_DATABASE_NAME, get_today, log_script_execution
from travel.rasp.info_center.info_center.suburban_notify.main import run_find_changes
from travel.rasp.info_center.info_center.suburban_notify.notify_params import NotifyParams

log = logging.getLogger(__name__)


def last_prepare_all():
    last_prepare_all_time = Timestamp.get('prepare_all')
    last_switch_time_str = maintenance.read_conf()['last_successful_switch']
    last_switch_time = datetime.strptime(last_switch_time_str, '%Y-%m-%d %H:%M:%S')
    data_change_time = max(last_switch_time, last_prepare_all_time)

    return MSK_TZ.localize(data_change_time).astimezone(UTC_TZ).replace(tzinfo=None)


def run():
    with lock(SUBS_CHANGES_LOCK, timeout=1700, database_name=SUBS_LOCK_DATABASE_NAME), \
            script_context('run_subscription_changes_after_prepare_all'), \
            log_script_execution():
        try:
            today = get_today()

            # Отправить изменения за сегодня.
            # Те, что уже отправлены и/или прошли - не отправятся
            run_find_changes(today, with_precache=True)

            # Отправить изменения на завтра.
            # Для тех точек, для которых прошло время сегодняшнего запуска скрипта по крону.
            run_find_changes(today + timedelta(days=1), minutes_from=18 * 60, with_precache=False)
        except OperationFailure as ex:
            log.exception('run_subscription_changes_after_prepare_all failed {}; {}'.format(ex.code, ex.details))
            raise
        except Exception:
            log.exception('run_subscription_changes_after_prepare_all failed')
            raise

        NotifyParams.set_param('last_successful_run_after_prepare_all', last_prepare_all())


if __name__ == '__main__':
    run()
