# coding: utf8
from __future__ import unicode_literals, absolute_import, division, print_function

import logging
from datetime import datetime

from common.models.schedule import RThreadType
from travel.rasp.info_center.info_center.suburban_notify.db import TRts


log = logging.getLogger(__name__)


class ChangeType(object):
    NOT_CHANGED = 'not_changed'
    CANCELLED = 'cancelled'
    CHANGED = 'changed'
    ADDED = 'added'
    NO_STOP = 'no_stop'


def get_rts_event_changes(event_tz, rel_event_tz, threads_diff):
    change = {'type': ChangeType.NOT_CHANGED}
    if event_tz is None:
        if rel_event_tz is not None:  # стала не начальной/конечной станцией
            change = {'type': ChangeType.ADDED}
    elif rel_event_tz is None:  # стала начальной/конечной станцией
        change = {'type': ChangeType.CANCELLED}
    else:
        event_diff = rel_event_tz - event_tz + threads_diff
        if event_diff:
            change = {'type': ChangeType.CHANGED, 'diff': event_diff}

    return change


def get_rts_changes(rts, rel_rts, event):
    change = {}

    # Нитка ходит в заданный день, но станция, присутствующая в базовой нитке, здесь отсутствует -
    # значит эта станция в этот день отменена.
    if not rel_rts:
        change['type'] = ChangeType.CANCELLED
        change['rts'] = rts
    elif rel_rts.is_no_stop():
        change['type'] = ChangeType.NO_STOP
        change['rts'] = rts
        change['rel_rts'] = rel_rts
    else:
        # Расчитываем, что таймзоны у базовой нитки и нитки-изменения совпадают.
        # Иначе придется делать много лишних вычислений про таймзоны.
        threads_diff = rel_rts.thread.get_start_time_minutes() - rts.thread.get_start_time_minutes()

        if event == 'departure':
            change = get_rts_event_changes(rts.tz_departure, rel_rts.tz_departure, threads_diff)
        elif event == 'arrival':
            change = get_rts_event_changes(rts.tz_arrival, rel_rts.tz_arrival, threads_diff)

        change['rts'] = rts
        if change['type'] != ChangeType.NOT_CHANGED:
            change['rel_rts'] = rel_rts

    return change


def get_segment_changes(rts_from, rts_to, rel_thread):

    # Ищем пару станций в rel_thread, соответствующих переданным rts.
    # TODO: Пока игнорируем возможность прохождения ниткой одной станции 2+ раза
    rel_rts_from, rel_rts_to = None, None

    # в нитке-отмене есть rts, но по факту они все отменены, поэтому их игнорируем
    if rel_thread.type != RThreadType.CANCEL_ID:
        for rel_rts in rel_thread.path:
            if rts_from.station == rel_rts.station:
                rel_rts_from = rel_rts
            elif rts_to.station == rel_rts.station:
                rel_rts_to = rel_rts

            if rel_rts_from and rel_rts_to:
                break

    change = {}
    change_from = get_rts_changes(rts_from, rel_rts_from, 'departure')
    if change_from:
        change['from'] = change_from

    change_to = get_rts_changes(rts_to, rel_rts_to, 'arrival')
    if change_to:
        change['to'] = change_to

    if change_from['type'] == ChangeType.NOT_CHANGED and change_to['type'] == ChangeType.NOT_CHANGED:
        change = {}

    return change


def get_basic_thread_changes(thread, thread_start_date, rts_from, rts_to):
    """
    Мы уже знаем, что по базовому календарю нитка должна ходить в заданный день.
    Надо понять, есть ли на самом деле изменения - для этого ищем изменяющие нитки, подходящие по дню старта,
    и ищем в них изменения для нашей пары станций.
    """

    change = None
    for rel_thread in thread.related_threads:
        if rel_thread.mask[thread_start_date]:  # есть изменения на заданный день
            if rel_thread.type == RThreadType.CANCEL_ID:
                segment_changes = get_segment_changes(rts_from, rts_to, rel_thread)
                change = {
                    'type': ChangeType.CANCELLED,
                    'basic_thread': thread,
                    'start_date': thread_start_date,
                    'rel_thread': rel_thread,
                }
                if segment_changes:
                    change.update(segment_changes)

            elif rel_thread.type == RThreadType.CHANGE_ID:
                segment_changes = get_segment_changes(rts_from, rts_to, rel_thread)
                if segment_changes:
                    change = {
                        'type': ChangeType.CHANGED,
                        'basic_thread': thread,
                        'start_date': thread_start_date,
                        'rel_thread': rel_thread,
                    }
                    change.update(segment_changes)

            break  # может быть только одно изменение в базе для конкретного дня

    return change


def get_new_segment_changes(thread_start_date, rts_from, rts_to):
    """
    Изменения для не-базовой нитки, для который нет базовой нитки в нашем интервале.
    Это могут быть либо нитки-изменения, либо нитки-назначения
    """

    thread = rts_from.thread
    change = {
        'rel_thread': thread,
        'type': ChangeType.ADDED,
        'start_date': thread_start_date,
        'from': {
            'type': ChangeType.ADDED,
            'rel_rts': rts_from,
        },
        'to': {
            'type': ChangeType.ADDED,
            'rel_rts': rts_to,
        }
    }

    return change


def get_start_date_for_departure_in_interval(rts, interval_from, interval_to, use_full_mask=True):
    # TODO: переход интервала за сутки не обрабатывается
    day_aware = interval_from.replace(hour=0, minute=0, second=0, microsecond=0)
    day = day_aware.astimezone(rts.thread.pytz).date()
    start_date = rts.get_start_date_for_event('departure', day, use_full_mask)
    if start_date:
        naive_start_dt = datetime.combine(start_date, rts.thread.tz_start_time)
        departure_dt = rts.get_departure_dt(naive_start_dt)

        # время отправления попадает в заданный интервал
        if interval_from <= departure_dt <= interval_to:
            return start_date


def get_changes_for_interval(interval_from, interval_to, segments):
    # Ищем базовые нитки, которые должны отправиться с заданной станции
    # в нужный день в нужный интервал времени по полному календарю.
    # Полный календарь (mask_full) составлен из дней хождения базовой нитки и всех ее изменений.
    changes = []
    basic_threads = set()
    for rts_from_id, rts_to_id in segments:
        rts_from = TRts.get(rts_from_id)
        thread = rts_from.thread
        if thread.type == RThreadType.BASIC_ID:
            # нитка стартует в такой день, чтобы отправиться с rts_from в нужный день
            start_date = get_start_date_for_departure_in_interval(rts_from, interval_from, interval_to)
            if start_date:
                basic_threads.add(thread)
                rts_to = TRts.get(rts_to_id)
                thread_changes = get_basic_thread_changes(thread, start_date, rts_from, rts_to)
                if thread_changes:
                    changes.append(thread_changes)

    # Если у изменяющей нитки не найдена базовая в нашем интервале,
    # то это новая нитка для интервала, которую не с чем сравнивать.
    basic_thread_ids = set(t.id for t in basic_threads)
    new_types = [RThreadType.CHANGE_ID, RThreadType.ASSIGNMENT_ID]
    for rts_from_id, rts_to_id in segments:
        rts_from = TRts.get(rts_from_id)
        thread = rts_from.thread
        if thread.basic_thread_id not in basic_thread_ids and thread.type in new_types:
            start_date = get_start_date_for_departure_in_interval(rts_from, interval_from, interval_to, False)
            if start_date:
                thread_changes = get_new_segment_changes(start_date, rts_from, TRts.get(rts_to_id))
                if thread_changes:
                    changes.append(thread_changes)

    return changes
