# coding: utf8
from __future__ import unicode_literals, absolute_import, division, print_function

import logging
from datetime import datetime, time

from travel.rasp.info_center.info_center.suburban_notify.changes.find import ChangeType
from travel.rasp.info_center.info_center.suburban_notify.utils import InitableSlots

log = logging.getLogger(__name__)


class SubscriptionChanges(InitableSlots):
    __slots__ = [
        'calc_date',
        'uid',
        'point_from_key',
        'point_to_key',
        'interval_to',
        'interval_from',
        'changes',  # Change

        'subscription',  # Subscription; не сериализуется
    ]

    def __init__(self, *args, **kwargs):
        self.changes = []
        super(SubscriptionChanges, self).__init__(*args, **kwargs)

    @classmethod
    def from_dict(cls, dct):
        """
        :param dct: created by serialize_sub_changes()
        :return: SubscriptionChanges object
        """
        dct = dict(dct)
        changes = dct.pop('changes', [])
        sub_changes = cls(**dct)
        for change_dict in changes:
            basic_thread_dict = change_dict.pop('basic_thread', None)
            rel_thread_dict = change_dict.pop('rel_thread', None)
            rts_from_dict = change_dict.pop('rts_from', None)
            rts_to_dict = change_dict.pop('rts_to', None)
            change_dict.pop('hash', None)

            change = Change(**change_dict)
            if basic_thread_dict:
                change.basic_thread = ThreadData(**basic_thread_dict)
            if rel_thread_dict:
                change.rel_thread = ThreadData(**rel_thread_dict)
            if rts_from_dict:
                change.rts_from = RTSChange(**rts_from_dict)
            if rts_to_dict:
                change.rts_to = RTSChange(**rts_to_dict)
            sub_changes.changes.append(change)

        return sub_changes

    def to_dict(self):
        res_dict = {
            'calc_date': self.calc_date,
            'uid': self.uid,
            'point_from_key': self.point_from_key,
            'point_to_key': self.point_to_key,
            'interval_to': self.interval_to,
            'interval_from': self.interval_from,
            'changes': []
        }
        for change in self.changes:
            res_dict['changes'].append(change.to_dict())

        return res_dict


class Change(InitableSlots):
    __slots__ = [
        'type',
        'start_date',
        'basic_thread',  # ThreadData
        'rel_thread',  # ThreadData
        'rts_from',  # RTSChange
        'rts_to',  # RTSChange
        'push_sent',
    ]

    slots_defaults = {
        'basic_thread': None,
        'rel_thread': None,
        'rts_from': None,
        'rts_to': None,
        'push_sent': False,
    }

    def get_eq_key(self):
        return (
            self.type,
            self.start_date,
            self.basic_thread.get_eq_key() if self.basic_thread else None,
            self.rel_thread.get_eq_key() if self.rel_thread else None,
            self.rts_from.get_eq_key() if self.rts_from else None,
            self.rts_to.get_eq_key() if self.rts_to else None,
        )

    def __hash__(self):
        return hash(self.get_eq_key())

    def __eq__(self, other):
        return hash(self) == hash(other)

    def to_dict(self):
        def obj_to_dict(obj):
            return obj.to_dict() if obj else None

        return {
            'type': self.type,
            'start_date': self.start_date,
            'basic_thread': obj_to_dict(self.basic_thread),
            'rel_thread': obj_to_dict(self.rel_thread),
            'rts_from': obj_to_dict(self.rts_from),
            'rts_to': obj_to_dict(self.rts_to),
            'push_sent': self.push_sent,
            'hash': self.__hash__()
        }

    def is_important(self):
        if self.type == ChangeType.CHANGED and self.rts_from and self.rts_to:
            not_important_from = 0 <= self.rts_from.diff <= 5
            not_important_to = -5 <= self.rts_to.diff <= 5
            return (not not_important_from) and (not not_important_to)

        return True

    def is_first_run_day(self):
        return self.rel_thread.is_first_run_day

    def sort_time(self):
        return (self.rts_from.schedule_time or self.rts_from.actual_time) if self.rts_from else None


class ThreadData(InitableSlots):
    __slots__ = [
        'key',
        'is_first_run_day',
        'uid',
        'number',
        'title',
    ]

    slots_defaults = {
        'is_first_run_day': None,
    }

    def get_eq_key(self):
        return self.key, self.is_first_run_day

    def __hash__(self):
        return hash(self.get_eq_key())

    def __eq__(self, other):
        return hash(self) == hash(other)

    def to_dict(self):
        return {
            'key': self.key,
            'is_first_run_day': self.is_first_run_day,
            'uid': self.uid,
            'number': self.number,
            'title': self.title,
        }


class RTSChange(InitableSlots):
    __slots__ = [
        'type',
        'station',
        'actual_time',
        'schedule_time',
        'diff',
        'first_station',
        'last_station',
    ]

    slots_defaults = {
        'first_station': None,
        'last_station': None,
        'actual_time': None,
        'schedule_time': None,
        'diff': None,
    }

    def get_eq_key(self):
        return (
            self.type,
            self.station,
            self.actual_time,
            self.schedule_time,
            self.diff,
            self.first_station,
            self.last_station,
        )

    def __hash__(self):
        return hash(self.get_eq_key())

    def __eq__(self, other):
        return hash(self) == hash(other)

    def to_dict(self):
        return {
            'type': self.type,
            'station': self.station,
            'actual_time': self.actual_time,
            'schedule_time': self.schedule_time,
            'diff': self.diff,
            'first_station': self.first_station,
            'last_station': self.last_station,
        }


def serialize_sub_changes(sub, changes, day):
    changes_prepared = serialize_changes(changes, day)

    sub_changes = {
        'calc_date': datetime.combine(day, time()),  # дата, на которую считались изменения
        # ключ
        'uid': sub.uid,
        'point_from_key': sub.point_from_key,
        'point_to_key': sub.point_to_key,
        'interval_from': sub.interval_from,
        'interval_to': sub.interval_to,

        # сами изменения
        'changes': changes_prepared,
    }

    return sub_changes


def serialize_changes(changes, day):
    changes_prepared = []

    for change in changes:
        try:
            prep_change = {
                'type': change['type'],
                'start_date': datetime.combine(change['start_date'], time())
            }

            basic_thread = change.get('basic_thread')
            if basic_thread:
                prep_change['basic_thread'] = {
                    'key': basic_thread.suburban_key,
                    'uid': basic_thread.uid,
                    'number': basic_thread.number,
                    'title': basic_thread.title,
                }

            rel_thread = change.get('rel_thread')
            if rel_thread:
                prep_change['rel_thread'] = {
                    'key': rel_thread.suburban_key,
                    'uid': rel_thread.uid,
                    'number': rel_thread.number,
                    'title': rel_thread.title,
                    'is_first_run_day': rel_thread.mask.is_first_run_day(day),
                }

            ch_from = change.get('from')
            if ch_from:
                prep_change['rts_from'] = serialize_rts_change(ch_from, 'departure', change['start_date'], change)

            ch_to = change.get('to')
            if ch_to:
                prep_change['rts_to'] = serialize_rts_change(ch_to, 'arrival', change['start_date'], change)

            changes_prepared.append(prep_change)
        except Exception:
            log.exception('Unexpected error on change serialization.')

    return changes_prepared


def serialize_rts_change(rts_change, event, start_date, change):
    prep_change = {
        'type': rts_change['type'],
    }

    def get_rts_event_dt(rts):
        return rts.get_event_dt_by_date(event, start_date, out_tz=rts.station.pytz).replace(tzinfo=None)

    if rts_change['type'] == ChangeType.CHANGED:
        rts = rts_change['rts']
        rel_rts = rts_change['rel_rts']

        event_dt = get_rts_event_dt(rts)
        event_dt_new = get_rts_event_dt(rel_rts)

        prep_change['schedule_time'] = event_dt
        prep_change['actual_time'] = event_dt_new
        # prep_change['diff'] = ((event_dt_new - event_dt).total_seconds()) // 60
        prep_change['diff'] = rts_change['diff']
        prep_change['station'] = rts.station.id
    elif rts_change['type'] == ChangeType.CANCELLED:
        rts = rts_change['rts']
        event_dt = get_rts_event_dt(rts)
        prep_change['schedule_time'] = event_dt
        prep_change['actual_time'] = None
        prep_change['last_station'] = change['rel_thread'].last_rts().station.id
        prep_change['first_station'] = change['rel_thread'].first_rts().station.id
        prep_change['station'] = rts.station.id
    elif rts_change['type'] == ChangeType.NO_STOP:
        rts = rts_change['rts']
        event_dt = get_rts_event_dt(rts)
        prep_change['schedule_time'] = event_dt
        prep_change['actual_time'] = None
        prep_change['station'] = rts.station.id
    elif rts_change['type'] == ChangeType.ADDED:
        rel_rts = rts_change['rel_rts']
        event_dt_new = get_rts_event_dt(rel_rts)
        prep_change['actual_time'] = event_dt_new
        prep_change['station'] = rel_rts.station.id
    elif rts_change['type'] == ChangeType.NOT_CHANGED:
        rts = rts_change['rts']
        event_dt = get_rts_event_dt(rts)
        prep_change['schedule_time'] = event_dt
        prep_change['station'] = rts.station.id

    return prep_change
