# coding: utf8
from __future__ import unicode_literals, absolute_import, division, print_function

import logging
from datetime import timedelta
from functools import partial

from travel.rasp.library.python.common23.logging import log_run_time
from travel.rasp.library.python.common23.date import environment
from travel.rasp.info_center.info_center.suburban_notify.changes.find import ChangeType
from travel.rasp.info_center.info_center.suburban_notify.db import TStation, load_points, get_point_by_key
from travel.rasp.info_center.info_center.suburban_notify.subscriptions.models import Frequency, Importance


log = logging.getLogger(__name__)
log_run_time = partial(log_run_time, logger=log)


def thread_title(thread, time_from):
    return '{:%H:%M} {}'.format(time_from, thread.title)


class TextGenerator(object):
    def get_text_for_change(self, change):
        title, description = None, None
        if change.type == ChangeType.CHANGED:
            ch_from, ch_to = change.rts_from.type, change.rts_to.type
            title = thread_title(change.basic_thread, change.rts_from.schedule_time)
            if ch_from == ChangeType.NOT_CHANGED and ch_to == ChangeType.CHANGED:
                description = 'отправится по расписанию в {:%H:%M}, прибудет в {} в {:%H:%M}'.format(
                    change.rts_from.schedule_time,
                    TStation.get(change.rts_to.station).title_ru,
                    change.rts_to.actual_time,
                )
            elif ch_from == ChangeType.CHANGED and ch_to == ChangeType.NOT_CHANGED:
                description = 'отправится в {:%H:%M}, прибудет в {} в {:%H:%M}'.format(
                    change.rts_from.actual_time,
                    TStation.get(change.rts_to.station).title_ru,
                    change.rts_to.schedule_time,
                )
            elif ch_from == ChangeType.CHANGED and ch_to == ChangeType.CHANGED:
                description = 'отправится в {:%H:%M}, прибудет в {} в {:%H:%M}'.format(
                    change.rts_from.actual_time,
                    TStation.get(change.rts_to.station).title_ru,
                    change.rts_to.actual_time,
                )
            elif ch_from == ChangeType.NOT_CHANGED and ch_to == ChangeType.NO_STOP:
                description = 'проследует станцию {} без остановки'.format(
                    TStation.get(change.rts_to.station).title_ru,
                )
            elif ch_from == ChangeType.CHANGED and ch_to == ChangeType.NO_STOP:
                description = 'отправится в {:%H:%M} и проследует станцию {} без остановки'.format(
                    change.rts_from.actual_time,
                    TStation.get(change.rts_to.station).title_ru,
                )
            elif ch_from == ChangeType.NO_STOP:
                description = 'проследует станцию {} без остановки, посадка невозможна'.format(
                    TStation.get(change.rts_from.station).title_ru,
                )
            elif ch_from == ChangeType.CANCELLED:
                description = 'отменен, будет следовать из {}'.format(
                    TStation.get(change.rts_from.first_station).title_ru,
                )
            elif ch_to == ChangeType.CANCELLED:
                description = 'проезд до станции {} невозможен, поезд следует до станции {}'.format(
                    TStation.get(change.rts_to.station).title_ru,
                    TStation.get(change.rts_to.last_station).title_ru,
                )
            else:
                print('AHTUNG: {} {}'.format(ch_from, ch_to))

        elif change.type == ChangeType.CANCELLED:
            title = thread_title(change.basic_thread, change.rts_from.schedule_time)
            description = 'отменен'
        elif change.type == ChangeType.ADDED:
            title = thread_title(change.rel_thread, change.rts_from.actual_time)
            description = 'будет отправляться в {:%H:%M} и прибывать в {:%H:%M}'.format(
                change.rts_from.actual_time,
                change.rts_to.actual_time,
            )

        return {'thread': title, 'description': description, 'full_text': '{} {}'.format(title, description)}

    def get_text_for_sub_changes(self, sub_changes, filtered_changes):
        sub_texts = []
        day_str = None

        for change in filtered_changes:
            text = self.get_text_for_change(change)['full_text']
            if text:
                sub_texts.append(text)
                if not day_str:  # день одинаковый для всех изменений
                    day_str = change.start_date.strftime('%d.%m')

        if not sub_texts:
            return None

        title = 'Изменения на ' + day_str
        if len(sub_texts) == 1:
            text = sub_texts[0]
        elif len(sub_texts) > 1:
            point_from = get_point_by_key(sub_changes.point_from_key)
            point_to = get_point_by_key(sub_changes.point_to_key)
            text = 'Изменения по направлению {} - {}'.format(point_from.L_title(), point_to.L_title())

        return {'title': title, 'text': text, 'texts': sub_texts}

    def load_objects(self, subs_changes):
        with log_run_time('load stations'):
            station_ids, point_keys = set(), set()
            for sub_changes in subs_changes:
                point_keys.add(sub_changes.point_from_key)
                point_keys.add(sub_changes.point_to_key)
                for change in sub_changes.changes:
                    if change.rts_from:
                        station_ids.add(change.rts_from.station)
                        station_ids.add(change.rts_from.first_station)
                        station_ids.add(change.rts_from.last_station)
                    if change.rts_to:
                        station_ids.add(change.rts_to.station)
                        station_ids.add(change.rts_to.first_station)
                        station_ids.add(change.rts_to.last_station)

            TStation.load_objs_by_ids(station_ids - {None})
            load_points(point_keys)

    def get_texts_for_changes(self, filtered_changes_by_sub):
        self.load_objects(filtered_changes_by_sub.keys())

        subs_texts = {}
        for sub_changes, filtered_changes in filtered_changes_by_sub.items():
            texts = self.get_text_for_sub_changes(sub_changes, filtered_changes)
            if texts:
                subs_texts[sub_changes] = texts

        return subs_texts


def filter_changes(sub_changes, only_new=True):
    changes = []

    for change in sub_changes.changes:
        if change.push_sent:
            continue

        if sub_changes.subscription.frequency == Frequency.FIRST_DAY and not change.is_first_run_day():
            continue

        if sub_changes.subscription.importance == Importance.ONLY_IMPORTANT and not change.is_important():
            continue

        if only_new:
            rts_from = change.rts_from
            if rts_from:
                station = TStation.get(rts_from.station)

                if rts_from.schedule_time:
                    event_time = station.pytz.localize(rts_from.schedule_time)
                else:
                    event_time = station.pytz.localize(rts_from.actual_time)

                if event_time < environment.now_aware() - timedelta(hours=1):
                    continue

        changes.append(change)

    return changes
