# coding: utf8
from __future__ import unicode_literals, absolute_import, division, print_function

import logging
from functools import partial

import socket

from travel.rasp.library.python.common23.date import environment
from travel.rasp.library.python.common23.logging import log_run_time
from travel.rasp.info_center.info_center.scripts.utils import update_one_script_log
from travel.rasp.info_center.info_center.suburban_notify.changes.changes_finder import ChangesFinder, ChangesFinderWithStorage
from travel.rasp.info_center.info_center.suburban_notify.changes.text import TextGenerator, filter_changes
from travel.rasp.info_center.info_center.suburban_notify.search import Searcher
from travel.rasp.info_center.info_center.suburban_notify.send_pushes_yt import run_send_pushes, YtSendPushes
from travel.rasp.info_center.info_center.suburban_notify.subscriptions.filters import get_subscriptions_filtered
from travel.rasp.info_center.info_center.suburban_notify.subscriptions.models import Subscription, Importance, Frequency
from travel.rasp.info_center.info_center.suburban_notify.utils import run_precache

log = logging.getLogger(__name__)
log_run_time = partial(log_run_time, logger=log)


def get_subscription_changes(
        point_from, point_to, interval_from, interval_to, day,
        importance=Importance.ANY, frequency=Frequency.EVERY_DAY,
        only_new=False,
        ):

    sub = Subscription(
        uid=None,
        point_from_key=point_from,
        point_to_key=point_to,
        interval_from=interval_from,
        interval_to=interval_to,
        importance=importance,
        frequency=frequency,
    )

    finder = ChangesFinder()
    subs_changes = finder.run([sub], day)
    sub_changes = subs_changes[0]

    text_gen = TextGenerator()
    text_gen.load_objects(subs_changes)

    changes_dict = sub_changes.to_dict()
    for i, ch in enumerate(sub_changes.changes):
        change = changes_dict['changes'][i]
        change['is_important'] = ch.is_important()
        change['is_first_run_day'] = ch.is_first_run_day()
        change['text'] = text_gen.get_text_for_change(ch)['full_text']

    filtered_changes = filter_changes(sub_changes, only_new=only_new)

    push_dict = YtSendPushes.make_pushes_dicts({sub_changes: filtered_changes}, text_gen=text_gen)
    if push_dict:
        push_dict = push_dict[0]

    if sub_changes:
        texts = text_gen.get_text_for_sub_changes(sub_changes, filtered_changes)
    else:
        texts = None

    return {
        'push': push_dict,
        'texts': texts,
        'changes': changes_dict,
    }


def find_changes_for_subs(subs, day, script_run_id=None, with_precache=True):
    if len(subs) > 30 and with_precache:
        run_precache()

    with log_run_time('load_searches'):
        searcher = Searcher()
        searcher.precalc_searches(subs)

    try:
        subs_stat = Subscription.get_subscriptions_stat()
    except Exception:
        log.exception("Can't get Subscription stats")
    else:
        log.info('Subscriptions: %s, users: %s', subs_stat['total'], subs_stat['users'])
        update_one_script_log({'subscriptions': {'count': subs_stat['total'], 'users': subs_stat['users']}})

    with log_run_time('find_and_save_changes'):
        finder = ChangesFinderWithStorage(script_run_id=script_run_id)
        subs_changes = finder.run(subs, day, searcher)

    update_one_script_log({'subs_changes': {'count': len(subs_changes)}})

    filtered_changes_by_sub = {}
    for sub_changes in subs_changes:
        filtered_changes_by_sub[sub_changes] = filter_changes(sub_changes)

    with log_run_time('send pushes for subs changes'):
        pushes_sent = run_send_pushes(filtered_changes_by_sub, script_run_id)
        if pushes_sent:
            with log_run_time('save all_sent'):
                finder.save_subs_changes_all_sent(filtered_changes_by_sub)


def run_find_changes(day, minutes_from=0, minutes_to=1440, limit=0, **kwargs):
    script_run_id = '{}__{}'.format(environment.now_utc().isoformat(), socket.gethostname())

    with log_run_time('get_subscriptions: {}: {} ({} - {})'.format(script_run_id, day, minutes_from, minutes_to)):
        subs = get_subscriptions_filtered(day, minutes_from, minutes_to, limit=limit)

    find_changes_for_subs(subs, day, script_run_id=script_run_id, **kwargs)


def run_find_changes_for_hour(day, hour, limit=0, **kwargs):
    """
    Запустить только для подписок, у которых в point_from сейчас hour времени.
    Например:
    run_find_changes_for_hour(datetime(2019, 1, 20), 19) - для точек, в которых сейчас 19 часов.

    Берем небольшой интервал около заданного часа, чтобы скрыть флуктуации часов и момента запуска скрипта.
    """
    hour_mins = hour * 60
    run_find_changes(day, minutes_from=hour_mins - 2, minutes_to=hour_mins + 5, limit=limit, **kwargs)
