# coding: utf8
from __future__ import unicode_literals, absolute_import, division, print_function

import json
import logging
import os
from collections import OrderedDict
from functools import partial
from urllib import urlencode

import yt.wrapper as yt
from django.conf import settings

from common.settings.configuration import Configuration
from common.settings.utils import define_setting
from travel.rasp.library.python.common23.logging import log_run_time
from common.apps.info_center.push import get_device_id_policy, get_install_id_policy
from common.data_api.sup.client import get_client, SupClient
from travel.rasp.library.python.api_clients.sup.yt2lb import Yt2Lb

from travel.rasp.info_center.info_center.scripts.utils import update_one_script_log
from travel.rasp.info_center.info_center.suburban_notify.changes.text import TextGenerator

log = logging.getLogger(__name__)
log_run_time = partial(log_run_time, logger=log)


YT_PROXY = 'hahn.yt.yandex.net'
define_setting('YT_PUSHES_PATH', default='//tmp', env={
    Configuration.TESTING: '//home/rasp/rasp/suburban_pushes/testing',
    Configuration.PRODUCTION: '//home/rasp/rasp/suburban_pushes/production',
})

define_setting(
    'PUSHES_LB_TOPIC',
    default='/sup/pushes-batch-test',
    env={Configuration.PRODUCTION: '/sup/pushes-batch'}
)
define_setting('PUSHES_LB_USER', default='robot-rasp')


class YtSendPushes(object):
    def __init__(self, pushes_dicts):
        self.pushes_dicts = pushes_dicts
        self.client = get_client()
        self.table = None
        self.pushes_count = None

    def write_pushes_yt(self, script_id):
        yt.update_config(
            {
                'token': settings.YT_TOKEN,
                'proxy': {'url': YT_PROXY}
            }
        )

        pushes_texts = [{'push': json.dumps(self.client.get_push_request(**push)[0])} for push in self.pushes_dicts]

        # yt.remove('//home/rasp/rasp/notification_pushes/testing/pushes_2019-02-26T11:08:25.245435')

        table_name = '{}'.format(script_id)
        self.table = os.path.join(settings.YT_PUSHES_PATH, table_name)

        # yt.create('table', self.table, recursive=True)

        update_one_script_log({'pushes': {'count': len(self.pushes_dicts), 'table_name': self.table}})

        with log_run_time('save {} pushes to yt table {}'.format(len(pushes_texts), self.table)):
            yt.write_table(self.table, pushes_texts, raw=False)

        log.info('run done')

    def send_pushes(self):
        """Отправка пушей через logbroker, RASPFRONT-9517"""
        yt2lb = Yt2Lb(
            lb_token=settings.LOGBROKER_OAUTH_TOKEN,
            yt_token=settings.YT_TOKEN,
            yt_proxy=YT_PROXY,
            yt_table=self.table,
            lb_topic=settings.PUSHES_LB_TOPIC,
            user=settings.PUSHES_LB_USER,
        )
        return yt2lb.run()

    @staticmethod
    def make_pushes_dicts(filtered_changes_by_sub, text_gen=None):
        pushes_dicts = []

        text_by_changes = (text_gen or TextGenerator()).get_texts_for_changes(filtered_changes_by_sub)
        log.info('Found %s subs with changes to send', len(text_by_changes))

        for sub_changes, texts in text_by_changes.items():
            data = OrderedDict([
                ('point_from_key', sub_changes.point_from_key),
                ('point_to_key', sub_changes.point_to_key)
            ])
            pushes_dicts.append({
                'project': 'suburban',
                'image': 'ic_notification',
                'receivers': [
                    '{} && uid=={}'.format(SupClient.get_app_id_tag(settings.SUBURBAN_APP_IDS), sub_changes.uid)
                ],
                'title': texts['title'],
                'text': texts['text'],
                'image_url': '',
                'url': 'yandextrains://subscriptionfeed/?{}'.format(urlencode(data)),
                'device_id_policy': get_device_id_policy('suburban_default_device_id'),
                'install_id_policy': get_install_id_policy(),
                'dry_run': False,
                'high_priority': True,
                'data': data
            })

        return pushes_dicts


def run_send_pushes(filtered_changes_by_sub, script_id):
    """
    sub_changes = SubscriptionChanges(uid=123, point_from_key='c213', point_to_key='c2')
    run_send_pushes([sub_changes], '2019-02-28 12:01')
    """
    pushes_dicts = YtSendPushes.make_pushes_dicts(filtered_changes_by_sub)
    if pushes_dicts:
        runner = YtSendPushes(pushes_dicts)
        runner.write_pushes_yt(script_id)
        response = runner.send_pushes()
        log.info('pushes sent %s', response)

    return True
