# coding: utf8
from __future__ import unicode_literals, absolute_import, division, print_function


import logging
from collections import defaultdict
from datetime import timedelta
from functools import partial

from common.models.schedule import RTStation, RThread, RThreadType
from common.models.transport import TransportType
from travel.rasp.library.python.common23.date import environment
from common.utils.date import RunMask
from travel.rasp.library.python.common23.logging import log_run_time
from common.utils.mysql_try_hard import mysql_try_hard
from travel.rasp.info_center.info_center.suburban_notify.db import parse_point_key
from travel.rasp.info_center.info_center.suburban_notify.subscriptions.models import Subscription
from travel.rasp.info_center.info_center.suburban_notify.utils import get_interval


log = logging.getLogger(__name__)
log_run_time = partial(log_run_time, logger=log)


def _load_from_points(subs):
    points_ids = defaultdict(set)
    for sub in subs:
        point_cls, point_id = parse_point_key(sub.point_from_key)
        points_ids[point_cls].add(point_id)

    points = []
    for point_cls, point_ids in points_ids.items():
        found_points, _ = point_cls.load_objs_by_ids(point_ids)
        points.extend(found_points)

    return points


def filter_subs_by_point_from_time(subs, minutes_from, minutes_to):
    with log_run_time('load_from_points'):
        from_points = _load_from_points(subs)

    filtered_points = []
    now = environment.now_aware()
    for point in from_points:
        interval_from, interval_to = get_interval(now.date(), minutes_from, minutes_to, point.pytz)
        if interval_from <= now <= interval_to:
            filtered_points.append(point)

    from_points_keys = set(point.get_key() for point in filtered_points)

    return [sub for sub in subs if sub.point_from_key in from_points_keys]


@mysql_try_hard
def _get_stations_with_possible_changes(day):
    days_to_search = [
        day + timedelta(days=-1),
        day,
        day + timedelta(days=1),
    ]
    mask = RunMask(days=days_to_search)

    possible_threads = []
    q = RThread.objects.filter(t_type_id=TransportType.SUBURBAN_ID).exclude(type_id=RThreadType.BASIC_ID)
    for thread_id, year_days, basic_thread_id in q.values_list('id', 'year_days', 'basic_thread_id'):
        thread_mask = RunMask(year_days)
        if thread_mask & mask:
            possible_threads.append(thread_id)
            possible_threads.append(basic_thread_id)

    station_ids_q = RTStation.objects.filter(thread_id__in=possible_threads).values_list('station_id', flat=True)
    stations_with_possible_changes = set(station_ids_q)

    return stations_with_possible_changes


def filter_subs_by_no_possible_changes(subs, day):
    """
    Фильтруем подписки, на которые не может быть изменений,
    т.к. нет небазовых ниток между заданными станциями.
    Подписки с городами не фильтруем.
    """
    stations = _get_stations_with_possible_changes(day)
    stations_s = {'s{}'.format(st_id) for st_id in stations}

    subs_filtered = []
    for sub in subs:
        only_cities = sub.point_from_key.startswith('c') and sub.point_to_key.startswith('c')
        has_possible_station = sub.point_from_key in stations_s or sub.point_to_key in stations_s
        if only_cities or has_possible_station:
            subs_filtered.append(sub)

    return subs_filtered


def get_subscriptions_filtered(day, minutes_from=0, minute_to=1440, limit=0):
    subs = Subscription.get_subscriptions(limit=limit)
    log.info('Subs unfiltered %s', len(subs))
    subs = filter_subs_by_point_from_time(subs, minutes_from, minute_to)
    log.info('Subs after filter by interval %s', len(subs))
    subs = filter_subs_by_no_possible_changes(subs, day)
    log.info('Subs after filter by "no possible changes" %s', len(subs))

    return subs
